#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
知识图谱相关接口定义
"""

from abc import ABC, abstractmethod
from typing import Dict, List, Any, Optional
from dataclasses import dataclass


@dataclass
class NodeInfo:
    """节点信息"""
    id: str
    name: str
    properties: Dict[str, Any]
    labels: List[str]


@dataclass
class GraphStats:
    """图谱统计信息"""
    node_count: int
    relationship_count: int
    node_labels: List[str]
    relationship_types: List[str]


class INodeRetriever(ABC):
    """节点检索接口"""
    
    @abstractmethod
    def get_all_nodes(self, limit: int = None) -> List[NodeInfo]:
        """获取所有节点"""
        pass
    
    @abstractmethod
    def get_nodes_by_label(self, label: str, limit: int = None) -> List[NodeInfo]:
        """根据标签获取节点"""
        pass
    
    @abstractmethod
    def get_node_by_id(self, node_id: str) -> Optional[NodeInfo]:
        """根据ID获取节点"""
        pass


class IKnowledgeGraphRetriever(ABC):
    """知识图谱检索接口"""
    
    @abstractmethod
    def initialize_graph(self, seed_nodes: List[str], max_depth: int, min_sitelinks: int) -> GraphStats:
        """初始化知识图谱"""
        pass
    
    @abstractmethod
    def get_graph_stats(self) -> GraphStats:
        """获取图谱统计信息"""
        pass
    
    @abstractmethod
    def clear_graph(self) -> bool:
        """清空图谱"""
        pass


class IKnowledgeGraphExporter(ABC):
    """知识图谱导出接口"""
    
    @abstractmethod
    def export_to_json(self, filename: str, domain: str) -> Dict[str, Any]:
        """导出为JSON格式"""
        pass
    
    @abstractmethod
    def export_with_apoc(self, filename: str) -> Dict[str, Any]:
        """使用APOC导出"""
        pass
