#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
模块1：Wikidata知识图谱检索与构建

该模块提供了从Wikidata检索和构建知识图谱的功能：
- 使用分类型查询策略：为四种关系类型分别生成独立的SPARQL查询
- 支持四种关系类型：instance_of(P31), subclass_of(P279), part_of(P361), has_part(P527)
- 支持递归查询，最大深度为3层（深度>3时自动回退到深度3）
- 支持Wikipedia存在性和多语言版本数量过滤
- 自动获取每个节点的Wikipedia英文描述并保存为节点属性
- 提供完整的数据处理流水线，确保语义准确性
"""

import logging
from typing import Dict, Any, Optional, Union, List
from ..utils.neo4j_client_full import Neo4jClient
from ..utils.file_utils import save_json
from ..utils.common_utils import ensure_dir
from ..utils.wikipedia_api import WikipediaAPI

logger = logging.getLogger(__name__)


class WikidataRetriever:
    """
    Wikidata知识图谱检索器
    
    提供从Wikidata检索指定领域知识图谱的完整功能，包括：
    - 分类型查询策略：为四种关系类型分别生成独立查询，确保语义准确性
    - 多种关系类型支持（instance_of, subclass_of, part_of, has_part）
    - 递归深度查询（支持深度1-3，深度>3时自动回退到深度3）
    - Wikipedia存在性和多语言版本过滤
    - 自动获取并保存Wikipedia英文描述为节点属性
    - 灵活的配置参数支持
    
    Attributes:
        neo4j_client: Neo4j数据库客户端
        
    Examples:
        >>> retriever = WikidataRetriever()
        >>> config = {
        ...     'domain': 'technology',
        ...     'entity_ids': ['Q11032', 'Q11053'],
        ...     'max_depth': 3,
        ...     'min_sitelinks': 5
        ... }
        >>> result = retriever.run_experiments(config)
        
    节点属性:
        - name: 实体的英文标签
        - wikipedia_summary: 实体的Wikipedia英文摘要
    """
    
    
    def __init__(self, neo4j_client=None, wikipedia_api: WikipediaAPI = None, 
                 max_retries: int = 5, retry_delay: float = 2.0):
        """
        初始化WikidataRetriever实例
        
        Args:
            neo4j_client: 可选的Neo4j客户端实例，如果不提供则创建新实例
            wikipedia_api: 可选的Wikipedia API实例
            max_retries: 最大重试次数，默认5次
            retry_delay: 重试延迟基数，默认2秒
        """
        if neo4j_client is not None:
            self.neo4j_client = neo4j_client
        else:
            # 使用默认配置创建Neo4j客户端
            self.neo4j_client = Neo4jClient(
                uri="bolt://localhost:7687",
                user="neo4j", 
                password="12345678"
            )
        
        if wikipedia_api is not None:
            self.wikipedia_api = wikipedia_api
        else:
            self.wikipedia_api = WikipediaAPI(
                max_retries=max_retries, 
                retry_delay=retry_delay
            )
        
        # 保存重试配置
        self.max_retries = max_retries
        self.retry_delay = retry_delay
    
    def _build_conservative_depth3_queries_by_type(self, wiki_nodes: List[str], min_sitelinks: int) -> List[str]:
        """为深度3构建分类型查询"""
        # 为每个起点节点单独构建CONSTRUCT和WHERE子句
        construct_parts = []
        where_parts = []
        
        for i, node in enumerate(wiki_nodes):
            # 为每个节点创建独特的变量名
            parent_label_var = f"?parentLabel{i}"
            parent_desc_var = f"?parentDescription{i}"
            
            # CONSTRUCT部分
            construct_parts.append(f"""
        wd:{node} a neo:node .
        wd:{node} neo:node {parent_label_var} .
        wd:{node} neo:description {parent_desc_var} .""")
            
            # WHERE部分
            where_parts.append(f"""
        wd:{node} rdfs:label {parent_label_var} .
        FILTER(LANG({parent_label_var}) = "en")
        
        OPTIONAL {{
            wd:{node} schema:description {parent_desc_var} .
            FILTER(LANG({parent_desc_var}) = "en")
        }}""")
        
        # 合并所有起点节点的构建部分
        all_parents_construct = "\n        ".join(construct_parts)
        all_parents_where = "\n        ".join(where_parts)
        
        # 构建VALUES子句
        wiki_nodes_values = " ".join([f"wd:{node}" for node in wiki_nodes])
        
        queries = []
        
        # 为每种关系类型创建单独的查询，直接使用语义化的关系名称
        # 根据关系类型在Wikidata中的频率分配不同的LIMIT
        relations = [
            ("P31", "neo:instance_of", 3000),    # instance_of 最常见
            ("P279", "neo:subclass_of", 3000),   # subclass_of 很常见  
            ("P361", "neo:part_of", 1000),       # part_of 较少见
            ("P527", "neo:has_part", 1000)       # has_part 较少见
        ]
        
        for prop, neo_rel, limit in relations:
            # 深度3查询：添加三层
            query = f"""PREFIX neo: <neo4j://voc#>
PREFIX schema: <http://schema.org/>
    CONSTRUCT {{
        {all_parents_construct}
        
        # 第一层
        ?child1 a neo:node .
        ?child1 neo:node ?childLabel1 .
        ?child1 neo:description ?childDescription1 .
        ?parent {neo_rel} ?child1 .
        
        # 第二层
        ?child2 a neo:node .
        ?child2 neo:node ?childLabel2 .
        ?child2 neo:description ?childDescription2 .
        ?child1 {neo_rel} ?child2 .
        
        # 第三层
        ?child3 a neo:node .
        ?child3 neo:node ?childLabel3 .
        ?child3 neo:description ?childDescription3 .
        ?child2 {neo_rel} ?child3 .
    }}
    WHERE {{
        {all_parents_where}
        
        VALUES ?parent {{ {wiki_nodes_values} }}

        # 第一层查询
        ?child1 wdt:{prop} ?parent .
        ?child1 rdfs:label ?childLabel1 .
        FILTER(LANG(?childLabel1) = "en")
        
        OPTIONAL {{
            ?child1 schema:description ?childDescription1 .
            FILTER(LANG(?childDescription1) = "en")
        }}
        
        FILTER EXISTS {{
            ?article1 schema:about ?child1 ;
                       schema:inLanguage "en" ;
                       schema:isPartOf <https://en.wikipedia.org/> .
        }}
        
        ?child1 wikibase:sitelinks ?sitelinks1 .
        FILTER(?sitelinks1 >= {min_sitelinks})
        
        # 第二层查询
        OPTIONAL {{
            ?child2 wdt:{prop} ?child1 .
            ?child2 rdfs:label ?childLabel2 .
            FILTER(LANG(?childLabel2) = "en")
            
            OPTIONAL {{
                ?child2 schema:description ?childDescription2 .
                FILTER(LANG(?childDescription2) = "en")
            }}
            
            FILTER EXISTS {{
                ?article2 schema:about ?child2 ;
                           schema:inLanguage "en" ;
                           schema:isPartOf <https://en.wikipedia.org/> .
            }}
            
            ?child2 wikibase:sitelinks ?sitelinks2 .
            FILTER(?sitelinks2 >= {min_sitelinks})
            
            # 第三层查询
            OPTIONAL {{
                ?child3 wdt:{prop} ?child2 .
                ?child3 rdfs:label ?childLabel3 .
                FILTER(LANG(?childLabel3) = "en")
                
                OPTIONAL {{
                    ?child3 schema:description ?childDescription3 .
                    FILTER(LANG(?childDescription3) = "en")
                }}
                
                FILTER EXISTS {{
                    ?article3 schema:about ?child3 ;
                               schema:inLanguage "en" ;
                               schema:isPartOf <https://en.wikipedia.org/> .
                }}
                
                ?child3 wikibase:sitelinks ?sitelinks3 .
                FILTER(?sitelinks3 >= {min_sitelinks})
            }}
        }}
    }}
    LIMIT {limit}"""
            queries.append(query)
        
        return queries

    def initialize_graph_mixed_semantic(self, wiki_nodes: Union[str, List[str]], depth: int = 3, min_sitelinks: int = 5) -> Dict[str, Any]:
        """
        使用分类型查询方法初始化知识图谱：为每种关系类型生成独立的SPARQL查询
        
        Args:
            wiki_nodes: Wikidata实体ID或ID列表，如 'Q11032' 或 ['Q11032', 'Q12136']
            depth: 递归查询深度，默认为3
            min_sitelinks: 最少Wikipedia语言版本数，默认为5
            
        Returns:
            Dict[str, Any]: 包含操作统计信息的字典
            
        Notes:
            核心策略：分类型查询，为四种关系类型(P31, P279, P361, P527)分别生成独立查询
            确保每种关系都有正确的语义名称，避免关系类型混合造成的语义模糊
            支持深度1-3，当depth>3时自动回退到深度3（完全等同于深度3实现）
        """
        # 确保wiki_nodes是列表格式用于日志显示
        if isinstance(wiki_nodes, str):
            display_nodes = [wiki_nodes]
        else:
            display_nodes = wiki_nodes
            
        logger.info(f"开始使用分类型查询方法初始化知识图谱，wiki_nodes={display_nodes}, depth={depth}, min_sitelinks={min_sitelinks}")
        logger.info("策略：分类型查询，为每种关系类型生成独立查询，确保语义准确性")
        
        stats = {}
        
        # 1. 清空数据库
        logger.info("清空现有数据库")
        self.neo4j_client.clear_database()
        stats['cleared_database'] = True
        
        # 2. 初始化n10s配置
        logger.info("初始化n10s RDF插件配置")
        self.neo4j_client.init_n10s_config()
        stats['n10s_initialized'] = True
        
        # 3. 构建分类型查询，为每种关系类型生成独立的SPARQL查询
        logger.info("构建分类型SPARQL查询")
        
        # 使用分类型查询，为四种关系类型分别生成查询
        sparql_queries = self._build_comprehensive_mixed_queries(display_nodes, depth, min_sitelinks)
        stats['sparql_queries_count'] = len(sparql_queries)
        
        # 执行每个分类型查询，确保完整的关系类型覆盖
        relation_names = ['P31(instance_of)', 'P279(subclass_of)', 'P361(part_of)', 'P527(has_part)']
        for i, query in enumerate(sparql_queries):
            if i < len(relation_names):
                rel_name = relation_names[i]
                logger.info(f"执行{rel_name}分类型查询...")
                self.neo4j_client.import_rdf_from_sparql(
                    query, 
                    max_retries=self.max_retries, 
                    retry_delay=self.retry_delay
                )
        
        stats['rdf_imported'] = True
        
        # 4. 数据后处理
        logger.info("重命名节点属性")
        self.neo4j_client.rename_node_properties()
        stats['properties_renamed'] = True
        
        logger.info("删除孤立节点")
        self.neo4j_client.remove_isolated_nodes()
        stats['isolated_nodes_removed'] = True
        
        # 5. 获取当前节点数量，为所有节点添加Wikipedia摘要
        node_count = self.neo4j_client.get_node_count()
        logger.info("添加Wikipedia摘要")
        enrich_stats = self.enrich_nodes_with_wikipedia_summaries(
            batch_size=10, 
            max_nodes=node_count  # 处理所有节点
        )
        stats['wikipedia_summary_stats'] = enrich_stats
        
        # 6. 收集统计信息
        relationship_count = self.neo4j_client.get_relationship_count()
        
        stats.update({
            'final_node_count': node_count,
            'final_relationship_count': relationship_count,
            'wiki_nodes': display_nodes,
            'depth': depth,
            'min_sitelinks': min_sitelinks,
            'layers_built': depth  # 记录构建的深度
        })
        
        logger.info(f"分类型知识图谱初始化完成，节点数: {node_count}, 关系数: {relationship_count}")
        return stats
    
    def _build_comprehensive_mixed_queries(self, wiki_nodes: List[str], depth: int, min_sitelinks: int) -> List[str]:
        """
        构建分类型查询：为每种关系类型生成独立的SPARQL查询
        
        策略：分类型查询方法，为四种关系类型(P31, P279, P361, P527)分别生成独立查询，
        确保每种关系都有正确的语义名称，避免关系混合造成的语义模糊
        """
        
        # 使用分类型查询，根据深度选择合适的查询方法
        return self._build_enhanced_mixed_type_queries(wiki_nodes, depth, min_sitelinks)
    
    def _build_enhanced_mixed_type_queries(self, wiki_nodes: List[str], depth: int, min_sitelinks: int) -> List[str]:
        """
        构建增强的分类型查询：根据深度路由到对应的分类型查询方法
        
        策略：统一的分类型查询路由，为每种关系类型生成独立查询
        支持深度1-3，当depth>3时自动回退到深度3（完全等同于深度3实现）
        """
        
        if depth == 1:
            return self._build_conservative_depth1_queries_by_type(wiki_nodes, min_sitelinks)
        elif depth == 2:
            return self._build_conservative_depth2_queries_by_type(wiki_nodes, min_sitelinks)
        elif depth == 3:
            return self._build_conservative_depth3_queries_by_type(wiki_nodes, min_sitelinks)
        else:
            # 深度>3时，直接回退到深度3，完全等同于深度3的实现
            logger.info(f"请求深度为{depth}，自动回退到深度3")
            return self._build_conservative_depth3_queries_by_type(wiki_nodes, min_sitelinks)
    
    def enrich_nodes_with_wikipedia_summaries(self, batch_size: int = 10, max_nodes: int = None) -> Dict[str, Any]:
        """
        为知识图谱节点添加Wikipedia摘要信息
        
        Args:
            batch_size: 每批处理的节点数量
            max_nodes: 最大处理节点数量，None表示处理所有节点
            
        Returns:
            Dict[str, Any]: 包含处理统计信息的字典
            
        Notes:
            该方法会：
            1. 获取没有Wikipedia摘要的节点
            2. 根据节点名称构造Wikipedia URL
            3. 调用Wikipedia API获取摘要
            4. 将摘要保存为节点的wikipedia_summary属性
        """
        logger.info("开始为节点添加Wikipedia摘要")
        
        stats = {
            'total_processed': 0,
            'successful_updates': 0,
            'failed_requests': 0,
            'not_found': 0,
            'errors': []
        }
        
        # 获取需要添加摘要的节点
        # 如果max_nodes为None，则获取所有没有摘要的节点
        if max_nodes is None:
            # 获取所有没有摘要的节点数量
            with self.neo4j_client.driver.session() as session:
                result = session.run("""
                    MATCH (n)
                    WHERE n.name IS NOT NULL AND n.name <> '' 
                    AND (n.wikipedia_summary IS NULL OR n.wikipedia_summary = '')
                    RETURN count(n) as count
                """)
                max_nodes = result.single()["count"]
        
        nodes_to_process = self.neo4j_client.get_nodes_without_wikipedia_summary(max_nodes)
        
        if not nodes_to_process:
            logger.info("所有节点都已有Wikipedia摘要")
            return stats
        
        logger.info(f"找到 {len(nodes_to_process)} 个需要添加摘要的节点")
        
        # 分批处理节点
        for i in range(0, len(nodes_to_process), batch_size):
            batch = nodes_to_process[i:i + batch_size]
            logger.info(f"处理第 {i//batch_size + 1} 批，共 {len(batch)} 个节点")
            
            for node_name in batch:
                try:
                    stats['total_processed'] += 1
                    
                    # 跳过空名称
                    if not node_name or not node_name.strip():
                        logger.warning("跳过空节点名称")
                        continue
                    
                    logger.debug(f"处理节点: {node_name}")
                    
                    # 获取Wikipedia摘要
                    summary = self.wikipedia_api.get_wikipedia_summary(node_name)
                    
                    if summary:
                        # 更新节点属性
                        updated_count = self.neo4j_client.add_wikipedia_summary_to_node(node_name, summary)
                        if updated_count > 0:
                            stats['successful_updates'] += 1
                            logger.info(f"成功为节点 '{node_name}' 添加Wikipedia摘要")
                        else:
                            logger.warning(f"节点 '{node_name}' 更新失败，可能不存在")
                    else:
                        stats['not_found'] += 1
                        logger.info(f"节点 '{node_name}' 未找到Wikipedia页面")
                        
                except Exception as e:
                    stats['failed_requests'] += 1
                    error_msg = f"处理节点 '{node_name}' 时出错: {str(e)}"
                    stats['errors'].append(error_msg)
                    logger.error(error_msg)
        
        # 记录最终统计
        logger.info(f"Wikipedia摘要添加完成:")
        logger.info(f"  总处理数: {stats['total_processed']}")
        logger.info(f"  成功更新: {stats['successful_updates']}")
        logger.info(f"  未找到页面: {stats['not_found']}")
        logger.info(f"  请求失败: {stats['failed_requests']}")
        
        return stats
    
    def _build_conservative_depth1_queries_by_type(self, wiki_nodes: List[str], min_sitelinks: int) -> List[str]:
        """为深度1构建分类型查询"""
        # 为每个起点节点单独构建CONSTRUCT和WHERE子句
        construct_parts = []
        where_parts = []
        
        for i, node in enumerate(wiki_nodes):
            # 为每个节点创建独特的变量名
            parent_label_var = f"?parentLabel{i}"
            parent_desc_var = f"?parentDescription{i}"
            
            # CONSTRUCT部分
            construct_parts.append(f"""
        wd:{node} a neo:node .
        wd:{node} neo:node {parent_label_var} .
        wd:{node} neo:description {parent_desc_var} .""")
            
            # WHERE部分
            where_parts.append(f"""
        wd:{node} rdfs:label {parent_label_var} .
        FILTER(LANG({parent_label_var}) = "en")
        
        OPTIONAL {{
            wd:{node} schema:description {parent_desc_var} .
            FILTER(LANG({parent_desc_var}) = "en")
        }}""")
        
        # 合并所有起点节点的构建部分
        all_parents_construct = "\n        ".join(construct_parts)
        all_parents_where = "\n        ".join(where_parts)
        
        # 构建VALUES子句
        wiki_nodes_values = " ".join([f"wd:{node}" for node in wiki_nodes])
        
        queries = []
        
        # 为每种关系类型创建单独的查询
        relations = [
            ("P31", "neo:instance_of", 2000),    # instance_of 最常见
            ("P279", "neo:subclass_of", 2000),   # subclass_of 很常见  
            ("P361", "neo:part_of", 1000),       # part_of 较少见
            ("P527", "neo:has_part", 1000)       # has_part 较少见
        ]
        
        for prop, neo_rel, limit in relations:
            # 深度1查询：仅添加一层
            query = f"""PREFIX neo: <neo4j://voc#>
PREFIX schema: <http://schema.org/>
    CONSTRUCT {{
        {all_parents_construct}
        
        # 第一层
        ?child1 a neo:node .
        ?child1 neo:node ?childLabel1 .
        ?child1 neo:description ?childDescription1 .
        ?parent {neo_rel} ?child1 .
    }}
    WHERE {{
        {all_parents_where}
        
        VALUES ?parent {{ {wiki_nodes_values} }}

        # 第一层查询
        ?child1 wdt:{prop} ?parent .
        ?child1 rdfs:label ?childLabel1 .
        FILTER(LANG(?childLabel1) = "en")
        
        OPTIONAL {{
            ?child1 schema:description ?childDescription1 .
            FILTER(LANG(?childDescription1) = "en")
        }}
        
        FILTER EXISTS {{
            ?article1 schema:about ?child1 ;
                       schema:inLanguage "en" ;
                       schema:isPartOf <https://en.wikipedia.org/> .
        }}
        
        ?child1 wikibase:sitelinks ?sitelinks1 .
        FILTER(?sitelinks1 >= {min_sitelinks})
    }}
    LIMIT {limit}"""
            queries.append(query)
        
        return queries
    
    def _build_conservative_depth2_queries_by_type(self, wiki_nodes: List[str], min_sitelinks: int) -> List[str]:
        """为深度2构建分类型查询"""
        # 为每个起点节点单独构建CONSTRUCT和WHERE子句
        construct_parts = []
        where_parts = []
        
        for i, node in enumerate(wiki_nodes):
            # 为每个节点创建独特的变量名
            parent_label_var = f"?parentLabel{i}"
            parent_desc_var = f"?parentDescription{i}"
            
            # CONSTRUCT部分
            construct_parts.append(f"""
        wd:{node} a neo:node .
        wd:{node} neo:node {parent_label_var} .
        wd:{node} neo:description {parent_desc_var} .""")
            
            # WHERE部分
            where_parts.append(f"""
        wd:{node} rdfs:label {parent_label_var} .
        FILTER(LANG({parent_label_var}) = "en")
        
        OPTIONAL {{
            wd:{node} schema:description {parent_desc_var} .
            FILTER(LANG({parent_desc_var}) = "en")
        }}""")
        
        # 合并所有起点节点的构建部分
        all_parents_construct = "\n        ".join(construct_parts)
        all_parents_where = "\n        ".join(where_parts)
        
        # 构建VALUES子句
        wiki_nodes_values = " ".join([f"wd:{node}" for node in wiki_nodes])
        
        queries = []
        
        # 为每种关系类型创建单独的查询
        relations = [
            ("P31", "neo:instance_of", 2500),    # instance_of 最常见
            ("P279", "neo:subclass_of", 2500),   # subclass_of 很常见  
            ("P361", "neo:part_of", 1000),       # part_of 较少见
            ("P527", "neo:has_part", 1000)       # has_part 较少见
        ]
        
        for prop, neo_rel, limit in relations:
            # 深度2查询：添加两层
            query = f"""PREFIX neo: <neo4j://voc#>
PREFIX schema: <http://schema.org/>
    CONSTRUCT {{
        {all_parents_construct}
        
        # 第一层
        ?child1 a neo:node .
        ?child1 neo:node ?childLabel1 .
        ?child1 neo:description ?childDescription1 .
        ?parent {neo_rel} ?child1 .
        
        # 第二层
        ?child2 a neo:node .
        ?child2 neo:node ?childLabel2 .
        ?child2 neo:description ?childDescription2 .
        ?child1 {neo_rel} ?child2 .
    }}
    WHERE {{
        {all_parents_where}
        
        VALUES ?parent {{ {wiki_nodes_values} }}

        # 第一层查询
        ?child1 wdt:{prop} ?parent .
        ?child1 rdfs:label ?childLabel1 .
        FILTER(LANG(?childLabel1) = "en")
        
        OPTIONAL {{
            ?child1 schema:description ?childDescription1 .
            FILTER(LANG(?childDescription1) = "en")
        }}
        
        FILTER EXISTS {{
            ?article1 schema:about ?child1 ;
                       schema:inLanguage "en" ;
                       schema:isPartOf <https://en.wikipedia.org/> .
        }}
        
        ?child1 wikibase:sitelinks ?sitelinks1 .
        FILTER(?sitelinks1 >= {min_sitelinks})
        
        # 第二层查询
        OPTIONAL {{
            ?child2 wdt:{prop} ?child1 .
            ?child2 rdfs:label ?childLabel2 .
            FILTER(LANG(?childLabel2) = "en")
            
            OPTIONAL {{
                ?child2 schema:description ?childDescription2 .
                FILTER(LANG(?childDescription2) = "en")
            }}
            
            FILTER EXISTS {{
                ?article2 schema:about ?child2 ;
                           schema:inLanguage "en" ;
                           schema:isPartOf <https://en.wikipedia.org/> .
            }}
            
            ?child2 wikibase:sitelinks ?sitelinks2 .
            FILTER(?sitelinks2 >= {min_sitelinks})
        }}
    }}
    LIMIT {limit}"""
            queries.append(query)
        
        return queries