#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
约束条件检查服务
"""

import logging
from ...core.interfaces.asr_interfaces import IConstraintChecker
from ...core.exceptions import ServiceError


class ConstraintCheckerService(IConstraintChecker):
    """约束条件检查服务"""
    
    def __init__(self, asr_evaluator):
        """
        初始化约束检查器
        
        Args:
            asr_evaluator: ASR评估器实例，用于计算相似度和PPL
        """
        self.asr_evaluator = asr_evaluator
        self.logger = logging.getLogger(__name__)
    
    def check_constraints(self, original_prompt: str, current_prompt: str, 
                         similarity_threshold: float = 0.7, 
                         ppl_threshold: float = 50.0) -> bool:
        """
        检查是否满足约束条件
        
        Args:
            original_prompt: 原始提示词
            current_prompt: 当前提示词
            similarity_threshold: 语义相似度阈值
            ppl_threshold: 困惑度阈值
            
        Returns:
            是否满足约束条件
        """
        try:
            similarity = self.asr_evaluator.calculate_semantic_similarity(original_prompt, current_prompt)
            ppl = self.asr_evaluator.calculate_ppl(current_prompt)
            
            return (similarity >= similarity_threshold and ppl <= ppl_threshold)
            
        except Exception as e:
            self.logger.error(f"约束条件检查失败: {e}")
            return False
