#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
知识图谱检索服务
专门处理知识图谱的构建和检索逻辑
"""

from typing import List, Dict, Any
import logging

from ...core.interfaces import IKnowledgeGraphRetriever, GraphStats
from ...core.exceptions import KnowledgeGraphError
from ...utils.neo4j_utils import Neo4jManager


class KnowledgeGraphRetriever(IKnowledgeGraphRetriever):
    """知识图谱检索服务"""
    
    def __init__(self, neo4j_manager: Neo4jManager, retry_config: dict = None):
        """
        初始化知识图谱检索器
        
        Args:
            neo4j_manager: Neo4j管理器
            retry_config: 重试配置字典，包含 max_retries, retry_delay 等参数
        """
        self.neo4j_manager = neo4j_manager
        self.logger = logging.getLogger(__name__)
        
        # 设置重试配置默认值
        if retry_config is None:
            retry_config = {
                'max_retries': 5,
                'retry_delay': 2.0,
                'wikipedia_timeout': 10
            }
        self.retry_config = retry_config
        
        # 导入可选依赖
        try:
            from ...modules.wikidata_retriever import WikidataRetriever
            # 使用重试配置初始化WikidataRetriever
            self.wikidata_retriever = WikidataRetriever(
                max_retries=retry_config.get('max_retries', 5),
                retry_delay=retry_config.get('retry_delay', 2.0)
            )
            self.has_wikidata = True
            self.logger.info(f"WikidataRetriever初始化成功，重试配置: {retry_config}")
        except ImportError:
            self.wikidata_retriever = None
            self.has_wikidata = False
            self.logger.warning("WikidataRetriever不可用，将使用Mock模式")
        except Exception as e:
            self.logger.error(f"WikidataRetriever初始化失败: {e}")
            self.wikidata_retriever = None
            self.has_wikidata = False
    
    def initialize_graph(self, seed_nodes: List[str], max_depth: int, min_sitelinks: int) -> GraphStats:
        """
        初始化知识图谱
        
        Args:
            seed_nodes: 种子节点列表
            max_depth: 最大检索深度
            min_sitelinks: 最小站点链接数
            
        Returns:
            GraphStats: 图谱统计信息
        """
        try:
            if self.has_wikidata:
                # 使用与测试脚本相同的混合语义方法，获得更多节点
                stats = self.wikidata_retriever.initialize_graph_mixed_semantic(
                    seed_nodes, max_depth, min_sitelinks
                )
                return self._convert_to_graph_stats(stats)
            else:
                # 使用Mock模式
                return self._initialize_mock_graph(seed_nodes)
                
        except Exception as e:
            raise KnowledgeGraphError(f"知识图谱初始化失败: {str(e)}")
    
    def get_graph_stats(self) -> GraphStats:
        """获取图谱统计信息"""
        try:
            stats = self.neo4j_manager.get_database_stats()
            return GraphStats(
                node_count=stats.get("node_count", 0),
                relationship_count=stats.get("relationship_count", 0),
                node_labels=stats.get("node_labels", []),
                relationship_types=stats.get("relationship_types", [])
            )
        except Exception as e:
            raise KnowledgeGraphError(f"获取图谱统计失败: {str(e)}")
    
    def clear_graph(self) -> bool:
        """清空图谱"""
        try:
            self.neo4j_manager.clear_database()
            return True
        except Exception as e:
            raise KnowledgeGraphError(f"清空图谱失败: {str(e)}")
    
    def _convert_to_graph_stats(self, stats: Dict[str, Any]) -> GraphStats:
        """转换统计信息格式"""
        return GraphStats(
            node_count=stats.get("final_node_count", 0),
            relationship_count=stats.get("final_relationship_count", 0),
            node_labels=stats.get("node_labels", []),
            relationship_types=stats.get("relationship_types", [])
        )
    
    def _initialize_mock_graph(self, seed_nodes: List[str]) -> GraphStats:
        """初始化Mock图谱"""
        self.logger.info("使用Mock模式创建知识图谱")
        
        try:
            # 创建一些基础Mock节点
            mock_nodes = [
                {"id": "Q1", "name": "Medicine", "type": "Domain"},
                {"id": "Q2", "name": "Disease", "type": "Category"},
                {"id": "Q3", "name": "Drug", "type": "Category"},
                {"id": "Q4", "name": "Treatment", "type": "Category"},
                {"id": "Q5", "name": "Symptom", "type": "Category"},
            ]
            
            # 创建节点
            for node_data in mock_nodes:
                create_query = """
                CREATE (n:Entity {
                    wikidata_id: $id,
                    name: $name,
                    type: $type,
                    created_at: timestamp()
                })
                """
                self.neo4j_manager.client.execute_query(create_query, node_data)
            
            # 创建关系
            relations = [
                ("Q2", "Q1", "PART_OF"),
                ("Q3", "Q1", "PART_OF"),
                ("Q4", "Q1", "PART_OF"),
                ("Q5", "Q2", "INDICATES"),
            ]
            
            for source, target, rel_type in relations:
                rel_query = f"""
                MATCH (a:Entity {{wikidata_id: $source}})
                MATCH (b:Entity {{wikidata_id: $target}})
                CREATE (a)-[r:{rel_type}]->(b)
                """
                self.neo4j_manager.client.execute_query(rel_query, {
                    "source": source, 
                    "target": target
                })
            
            # 返回Mock统计
            return GraphStats(
                node_count=len(mock_nodes),
                relationship_count=len(relations),
                node_labels=["Entity"],
                relationship_types=["PART_OF", "INDICATES"]
            )
            
        except Exception as e:
            raise KnowledgeGraphError(f"Mock图谱创建失败: {str(e)}")
