#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
节点服务
专门处理节点的查询和管理
"""

from typing import List, Optional
import logging

from ...core.interfaces import INodeRetriever, NodeInfo
from ...core.exceptions import KnowledgeGraphError
from ...utils.neo4j_utils import Neo4jManager


class NodeService(INodeRetriever):
    """节点查询服务"""
    
    def __init__(self, neo4j_manager: Neo4jManager):
        """
        初始化节点服务
        
        Args:
            neo4j_manager: Neo4j管理器
        """
        self.neo4j_manager = neo4j_manager
        self.logger = logging.getLogger(__name__)
    
    def get_all_nodes(self, limit: int = None) -> List[NodeInfo]:
        """
        获取所有节点
        
        Args:
            limit: 限制数量
            
        Returns:
            节点信息列表
        """
        try:
            query = "MATCH (n) RETURN id(n) as id, labels(n) as labels, properties(n) as properties"
            if limit:
                query += f" LIMIT {limit}"
            
            result = self.neo4j_manager.client.execute_query(query)
            
            nodes = []
            for record in result:
                node_info = self._create_node_info(record)
                if node_info:
                    nodes.append(node_info)
            
            return nodes
            
        except Exception as e:
            raise KnowledgeGraphError(f"获取所有节点失败: {str(e)}")
    
    def get_nodes_by_label(self, label: str, limit: int = None) -> List[NodeInfo]:
        """
        根据标签获取节点
        
        Args:
            label: 节点标签
            limit: 限制数量
            
        Returns:
            节点信息列表
        """
        try:
            query = f"""
            MATCH (n:{label}) 
            RETURN id(n) as id, labels(n) as labels, properties(n) as properties
            """
            if limit:
                query += f" LIMIT {limit}"
            
            result = self.neo4j_manager.client.execute_query(query)
            
            nodes = []
            for record in result:
                node_info = self._create_node_info(record)
                if node_info:
                    nodes.append(node_info)
            
            return nodes
            
        except Exception as e:
            raise KnowledgeGraphError(f"根据标签获取节点失败: {str(e)}")
    
    def get_node_by_id(self, node_id: str) -> Optional[NodeInfo]:
        """
        根据ID获取节点
        
        Args:
            node_id: 节点ID
            
        Returns:
            节点信息
        """
        try:
            # 尝试作为内部ID查询
            try:
                query = """
                MATCH (n) WHERE id(n) = $node_id
                RETURN id(n) as id, labels(n) as labels, properties(n) as properties
                """
                result = self.neo4j_manager.client.execute_query(query, {"node_id": int(node_id)})
            except ValueError:
                # 作为属性查询
                query = """
                MATCH (n) WHERE n.wikidata_id = $node_id OR n.id = $node_id
                RETURN id(n) as id, labels(n) as labels, properties(n) as properties
                """
                result = self.neo4j_manager.client.execute_query(query, {"node_id": node_id})
            
            for record in result:
                return self._create_node_info(record)
            
            return None
            
        except Exception as e:
            raise KnowledgeGraphError(f"根据ID获取节点失败: {str(e)}")
    
    def _create_node_info(self, record) -> Optional[NodeInfo]:
        """
        从查询记录创建节点信息
        
        Args:
            record: 查询记录
            
        Returns:
            节点信息对象
        """
        try:
            properties = dict(record["properties"])
            labels = list(record["labels"])
            
            # 获取节点名称
            name = (properties.get('name') or 
                   properties.get('wikidata_id') or 
                   f"node_{record['id']}")
            
            return NodeInfo(
                id=str(record["id"]),
                name=name,
                properties=properties,
                labels=labels
            )
            
        except Exception as e:
            self.logger.warning(f"创建节点信息失败: {e}")
            return None
