#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
文件操作工具模块
提供统一的文件读写和管理功能
"""

import json
import csv
import os
import shutil
from pathlib import Path
from typing import Any, Dict, List, Optional, Union
from datetime import datetime
import pandas as pd

def ensure_directory(path: Union[str, Path]) -> Path:
    """
    确保目录存在，如果不存在则创建
    
    Args:
        path: 目录路径
        
    Returns:
        Path对象
    """
    path = Path(path)
    path.mkdir(parents=True, exist_ok=True)
    return path

def get_timestamped_filename(base_name: str, extension: str, include_date: bool = True) -> str:
    """
    生成带时间戳的文件名
    
    Args:
        base_name: 基础文件名
        extension: 文件扩展名
        include_date: 是否包含日期
        
    Returns:
        带时间戳的文件名
    """
    if include_date:
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    else:
        timestamp = datetime.now().strftime("%H%M%S")
    
    return f"{base_name}_{timestamp}.{extension}"

def save_json(data: Any, file_path: Union[str, Path], ensure_dir: bool = True) -> None:
    """
    保存数据为JSON文件
    
    Args:
        data: 要保存的数据
        file_path: 文件路径
        ensure_dir: 是否确保目录存在
    """
    file_path = Path(file_path)
    
    if ensure_dir:
        ensure_directory(file_path.parent)
    
    with open(file_path, 'w', encoding='utf-8') as f:
        json.dump(data, f, ensure_ascii=False, indent=2)

def load_json(file_path: Union[str, Path]) -> Any:
    """
    从JSON文件加载数据
    
    Args:
        file_path: 文件路径
        
    Returns:
        加载的数据
    """
    file_path = Path(file_path)
    
    if not file_path.exists():
        raise FileNotFoundError(f"文件不存在: {file_path}")
    
    with open(file_path, 'r', encoding='utf-8') as f:
        return json.load(f)

def save_csv(data: Union[List[Dict], pd.DataFrame], file_path: Union[str, Path], ensure_dir: bool = True) -> None:
    """
    保存数据为CSV文件
    
    Args:
        data: 要保存的数据（字典列表或DataFrame）
        file_path: 文件路径  
        ensure_dir: 是否确保目录存在
    """
    file_path = Path(file_path)
    
    if ensure_dir:
        ensure_directory(file_path.parent)
    
    if isinstance(data, pd.DataFrame):
        data.to_csv(file_path, index=False, encoding='utf-8')
    elif isinstance(data, list) and data and isinstance(data[0], dict):
        df = pd.DataFrame(data)
        df.to_csv(file_path, index=False, encoding='utf-8')
    else:
        raise ValueError("数据格式不支持，请提供DataFrame或字典列表")

def load_csv(file_path: Union[str, Path]) -> pd.DataFrame:
    """
    从CSV文件加载数据
    
    Args:
        file_path: 文件路径
        
    Returns:
        DataFrame对象
    """
    file_path = Path(file_path)
    
    if not file_path.exists():
        raise FileNotFoundError(f"文件不存在: {file_path}")
    
    return pd.read_csv(file_path, encoding='utf-8')

def backup_file(file_path: Union[str, Path], backup_dir: Optional[Union[str, Path]] = None) -> Path:
    """
    备份文件
    
    Args:
        file_path: 要备份的文件路径
        backup_dir: 备份目录，如果为None则在原目录下创建backup子目录
        
    Returns:
        备份文件路径
    """
    file_path = Path(file_path)
    
    if not file_path.exists():
        raise FileNotFoundError(f"文件不存在: {file_path}")
    
    if backup_dir is None:
        backup_dir = file_path.parent / "backup"
    else:
        backup_dir = Path(backup_dir)
    
    ensure_directory(backup_dir)
    
    # 生成备份文件名
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    backup_name = f"{file_path.stem}_{timestamp}{file_path.suffix}"
    backup_path = backup_dir / backup_name
    
    # 复制文件
    shutil.copy2(file_path, backup_path)
    
    return backup_path

def cleanup_old_files(directory: Union[str, Path], pattern: str = "*", max_age_days: int = 30) -> List[Path]:
    """
    清理旧文件
    
    Args:
        directory: 目录路径
        pattern: 文件模式
        max_age_days: 最大保留天数
        
    Returns:
        被删除的文件列表
    """
    directory = Path(directory)
    
    if not directory.exists():
        return []
    
    deleted_files = []
    current_time = datetime.now()
    
    for file_path in directory.glob(pattern):
        if file_path.is_file():
            file_age = current_time - datetime.fromtimestamp(file_path.stat().st_mtime)
            if file_age.days > max_age_days:
                file_path.unlink()
                deleted_files.append(file_path)
    
    return deleted_files

def get_file_size(file_path: Union[str, Path]) -> int:
    """
    获取文件大小（字节）
    
    Args:
        file_path: 文件路径
        
    Returns:
        文件大小
    """
    file_path = Path(file_path)
    
    if not file_path.exists():
        raise FileNotFoundError(f"文件不存在: {file_path}")
    
    return file_path.stat().st_size

def format_file_size(size_bytes: int) -> str:
    """
    格式化文件大小
    
    Args:
        size_bytes: 字节数
        
    Returns:
        格式化的文件大小字符串
    """
    if size_bytes == 0:
        return "0B"
    
    size_names = ["B", "KB", "MB", "GB", "TB"]
    i = 0
    while size_bytes >= 1024 and i < len(size_names) - 1:
        size_bytes /= 1024.0
        i += 1
    
    return f"{size_bytes:.1f}{size_names[i]}"

class FileManager:
    """文件管理器类"""
    
    def __init__(self, base_directory: Union[str, Path]):
        """
        初始化文件管理器
        
        Args:
            base_directory: 基础目录
        """
        self.base_directory = Path(base_directory)
        ensure_directory(self.base_directory)
    
    def get_path(self, *path_parts: str) -> Path:
        """
        获取相对于基础目录的路径
        
        Args:
            *path_parts: 路径组件
            
        Returns:
            完整路径
        """
        return self.base_directory.joinpath(*path_parts)
    
    def save_data(self, data: Any, filename: str, format: str = "json") -> Path:
        """
        保存数据
        
        Args:
            data: 要保存的数据
            filename: 文件名
            format: 文件格式（json或csv）
            
        Returns:
            保存的文件路径
        """
        if format == "json":
            file_path = self.get_path(f"{filename}.json")
            save_json(data, file_path)
        elif format == "csv":
            file_path = self.get_path(f"{filename}.csv")
            save_csv(data, file_path)
        else:
            raise ValueError(f"不支持的格式: {format}")
        
        return file_path
    
    def load_data(self, filename: str, format: str = "json") -> Any:
        """
        加载数据
        
        Args:
            filename: 文件名（不含扩展名）
            format: 文件格式（json或csv）
            
        Returns:
            加载的数据
        """
        if format == "json":
            file_path = self.get_path(f"{filename}.json")
            return load_json(file_path)
        elif format == "csv":
            file_path = self.get_path(f"{filename}.csv")
            return load_csv(file_path)
        else:
            raise ValueError(f"不支持的格式: {format}")
    
    def list_files(self, pattern: str = "*") -> List[Path]:
        """
        列出文件
        
        Args:
            pattern: 文件模式
            
        Returns:
            文件列表
        """
        return list(self.base_directory.glob(pattern))
    
    def cleanup(self, max_age_days: int = 30) -> List[Path]:
        """
        清理旧文件
        
        Args:
            max_age_days: 最大保留天数
            
        Returns:
            被删除的文件列表
        """
        return cleanup_old_files(self.base_directory, max_age_days=max_age_days)

if __name__ == "__main__":
    # 测试文件操作功能
    import tempfile
    
    # 创建临时目录进行测试
    with tempfile.TemporaryDirectory() as temp_dir:
        print(f"测试目录: {temp_dir}")
        
        # 测试文件管理器
        fm = FileManager(temp_dir)
        
        # 测试保存和加载JSON
        test_data = {"name": "test", "value": 123, "items": [1, 2, 3]}
        json_path = fm.save_data(test_data, "test_data", "json")
        print(f"保存JSON: {json_path}")
        
        loaded_data = fm.load_data("test_data", "json")
        print(f"加载JSON: {loaded_data}")
        
        # 测试保存和加载CSV
        csv_data = [
            {"name": "Alice", "age": 25},
            {"name": "Bob", "age": 30}
        ]
        csv_path = fm.save_data(csv_data, "test_data", "csv")
        print(f"保存CSV: {csv_path}")
        
        loaded_csv = fm.load_data("test_data", "csv")
        print(f"加载CSV: {loaded_csv}")
        
        # 测试列出文件
        files = fm.list_files()
        print(f"文件列表: {files}")
        
        print("文件操作测试完成")
