#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
日志工具模块
提供统一的日志管理功能
"""

import logging
import os
from pathlib import Path
from datetime import datetime
from typing import Optional

def setup_logger(
    name: str,
    log_file: Optional[str] = None,
    level: str = "INFO",
    log_dir: Optional[str] = None,
    enable_file_logging: bool = False
) -> logging.Logger:
    """
    设置日志记录器
    
    Args:
        name: 日志记录器名称
        log_file: 日志文件名，如果为None则使用默认命名
        level: 日志级别
        log_dir: 日志目录，如果为None则使用默认目录
        enable_file_logging: 是否启用文件日志
        
    Returns:
        配置好的日志记录器
    """
    # 创建日志记录器
    logger = logging.getLogger(name)
    logger.setLevel(getattr(logging, level.upper()))
    
    # 避免重复添加处理器
    if logger.handlers:
        logger.handlers.clear()
    
    # 创建格式器
    formatter = logging.Formatter(
        '%(asctime)s - %(name)s - %(levelname)s - %(message)s'
    )
    
    # 只有明确启用文件日志时才创建文件
    if enable_file_logging:
        # 确定日志目录
        if log_dir is None:
            log_dir = Path(__file__).parent.parent.parent / "logs"
        else:
            log_dir = Path(log_dir)
        
        log_dir.mkdir(parents=True, exist_ok=True)
        
        # 确定日志文件名
        if log_file is None:
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            log_file = f"{name}_{timestamp}.log"
        
        log_path = log_dir / log_file
        
        # 文件处理器
        file_handler = logging.FileHandler(log_path, encoding='utf-8')
        file_handler.setLevel(getattr(logging, level.upper()))
        file_handler.setFormatter(formatter)
        logger.addHandler(file_handler)
    
    # 控制台处理器 (总是添加)
    console_handler = logging.StreamHandler()
    console_handler.setLevel(getattr(logging, level.upper()))
    console_handler.setFormatter(formatter)
    logger.addHandler(console_handler)
    
    return logger

def get_logger(name: str) -> logging.Logger:
    """
    获取已配置的日志记录器
    
    Args:
        name: 日志记录器名称
        
    Returns:
        日志记录器
    """
    logger = logging.getLogger(name)
    if not logger.handlers:
        # 如果没有处理器，使用默认设置
        return setup_logger(name)
    return logger

def log_function_call(func):
    """
    装饰器：记录函数调用
    """
    def wrapper(*args, **kwargs):
        logger = get_logger(func.__module__)
        logger.info(f"调用函数: {func.__name__}")
        try:
            result = func(*args, **kwargs)
            logger.info(f"函数 {func.__name__} 执行成功")
            return result
        except Exception as e:
            logger.error(f"函数 {func.__name__} 执行失败: {e}")
            raise
    return wrapper

def log_execution_time(func):
    """
    装饰器：记录函数执行时间
    """
    import time
    
    def wrapper(*args, **kwargs):
        logger = get_logger(func.__module__)
        start_time = time.time()
        
        try:
            result = func(*args, **kwargs)
            end_time = time.time()
            execution_time = end_time - start_time
            logger.info(f"函数 {func.__name__} 执行时间: {execution_time:.2f}秒")
            return result
        except Exception as e:
            end_time = time.time()
            execution_time = end_time - start_time
            logger.error(f"函数 {func.__name__} 执行失败 (耗时 {execution_time:.2f}秒): {e}")
            raise
            
    return wrapper

class PipelineLogger:
    """Pipeline专用日志记录器"""
    
    def __init__(self, pipeline_name: str, domain: str, stage: str, enable_file_logging: bool = True):
        self.pipeline_name = pipeline_name
        self.domain = domain 
        self.stage = stage
        self.logger_name = f"{pipeline_name}_{domain}_{stage}"
        self.logger = setup_logger(self.logger_name, enable_file_logging=enable_file_logging)
        
    def info(self, message: str):
        """记录信息日志"""
        self.logger.info(f"[{self.domain}|{self.stage}] {message}")
        
    def error(self, message: str):
        """记录错误日志"""
        self.logger.error(f"[{self.domain}|{self.stage}] {message}")
        
    def warning(self, message: str):
        """记录警告日志"""
        self.logger.warning(f"[{self.domain}|{self.stage}] {message}")
        
    def debug(self, message: str):
        """记录调试日志"""
        self.logger.debug(f"[{self.domain}|{self.stage}] {message}")
        
    def log_stage_start(self):
        """记录阶段开始"""
        self.info(f"=== 开始执行 {self.stage} 阶段 ===")
        
    def log_stage_end(self, success: bool = True):
        """记录阶段结束"""
        status = "成功" if success else "失败"
        self.info(f"=== {self.stage} 阶段执行{status} ===")
        
    def log_step(self, step_name: str, message: str):
        """记录步骤信息"""
        self.info(f"[步骤: {step_name}] {message}")

if __name__ == "__main__":
    # 测试日志功能
    logger = setup_logger("test")
    logger.info("这是一条信息日志")
    logger.warning("这是一条警告日志")
    logger.error("这是一条错误日志")
    
    # 测试Pipeline日志
    pipeline_logger = PipelineLogger("test_pipeline", "medicine", "stage1")
    pipeline_logger.log_stage_start()
    pipeline_logger.log_step("知识图谱检索", "开始检索医疗领域知识图谱")
    pipeline_logger.info("测试Pipeline日志功能")
    pipeline_logger.log_stage_end(True)
