#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Neo4j连接工厂
提供统一的Neo4j连接管理，符合工程化标准
"""

from typing import Optional, Dict, Any
from .neo4j_utils import Neo4jManager, Neo4jConnectionConfig
from .config_manager import get_config_manager
from .logger_utils import get_logger

logger = get_logger(__name__)

class Neo4jConnectionFactory:
    """Neo4j连接工厂，单例模式"""
    
    _instance: Optional['Neo4jConnectionFactory'] = None
    _manager: Optional[Neo4jManager] = None
    
    def __new__(cls):
        if cls._instance is None:
            cls._instance = super().__new__(cls)
        return cls._instance
    
    def __init__(self):
        if not hasattr(self, '_initialized'):
            self._initialized = True
            self._manager = None
    
    def get_connection(self) -> Neo4jManager:
        """
        获取Neo4j连接管理器
        
        Returns:
            Neo4j管理器实例
        """
        if self._manager is None or not self._manager.is_connected():
            self._create_connection()
        
        return self._manager
    
    def _create_connection(self) -> None:
        """创建Neo4j连接"""
        try:
            # 从配置管理器获取数据库配置
            config_manager = get_config_manager()
            db_config = config_manager.get_database_config()
            
            # 创建连接配置
            neo4j_config = Neo4jConnectionConfig(
                uri=db_config.neo4j_uri,
                user=db_config.neo4j_user,
                password=db_config.neo4j_password
            )
            
            # 创建管理器并连接
            self._manager = Neo4jManager(neo4j_config)
            
            if self._manager.connect():
                logger.info("Neo4j连接工厂：连接创建成功")
            else:
                logger.error("Neo4j连接工厂：连接创建失败")
                
        except Exception as e:
            logger.error(f"Neo4j连接工厂：创建连接时出错 - {e}")
            self._manager = None
    
    def test_connection(self) -> bool:
        """
        测试连接状态
        
        Returns:
            连接是否正常
        """
        try:
            manager = self.get_connection()
            if manager and manager.is_connected():
                return manager.test_connection()
            return False
        except Exception as e:
            logger.error(f"Neo4j连接工厂：测试连接失败 - {e}")
            return False
    
    def get_database_stats(self) -> Dict[str, Any]:
        """
        获取数据库统计信息
        
        Returns:
            数据库统计字典
        """
        try:
            manager = self.get_connection()
            if manager and manager.is_connected():
                return manager.get_database_stats()
            return {}
        except Exception as e:
            logger.error(f"Neo4j连接工厂：获取数据库统计失败 - {e}")
            return {}
    
    def close_connection(self) -> None:
        """关闭连接"""
        if self._manager:
            self._manager.disconnect()
            self._manager = None
            logger.info("Neo4j连接工厂：连接已关闭")

# 全局连接工厂实例
_connection_factory = Neo4jConnectionFactory()

def get_neo4j_connection() -> Neo4jManager:
    """
    获取Neo4j连接的统一入口
    
    Returns:
        Neo4j管理器实例
    """
    return _connection_factory.get_connection()

def test_neo4j_connection() -> bool:
    """
    测试Neo4j连接的统一入口
    
    Returns:
        连接是否正常
    """
    return _connection_factory.test_connection()

def get_neo4j_stats() -> Dict[str, Any]:
    """
    获取Neo4j数据库统计的统一入口
    
    Returns:
        数据库统计字典
    """
    return _connection_factory.get_database_stats()

def close_neo4j_connection() -> None:
    """关闭Neo4j连接的统一入口"""
    _connection_factory.close_connection()

if __name__ == "__main__":
    # 测试连接工厂
    print("测试Neo4j连接工厂...")
    
    # 测试连接
    if test_neo4j_connection():
        print("✅ Neo4j连接测试成功")
        
        # 获取统计信息
        stats = get_neo4j_stats()
        print(f"📊 数据库统计: {stats}")
        
        # 获取连接实例
        manager = get_neo4j_connection()
        print(f"🔗 连接管理器: {manager}")
        
    else:
        print("❌ Neo4j连接测试失败")
    
    # 关闭连接
    close_neo4j_connection()
    print("🔒 连接已关闭")
