#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
随机数种子设置工具
确保实验的可重现性
"""

import random
import os
from typing import Optional
import logging

logger = logging.getLogger(__name__)


def set_random_seed(seed: Optional[int] = None, enable_deterministic: bool = True) -> int:
    """
    设置全局随机数种子，确保实验可重现性
    
    Args:
        seed: 随机数种子，如果为None则使用默认值42
        enable_deterministic: 是否启用确定性模式
        
    Returns:
        int: 实际使用的随机数种子
    """
    if seed is None:
        seed = 42
    
    # 设置Python内置random模块种子
    random.seed(seed)
    
    # 设置环境变量（如果需要的话）
    os.environ['PYTHONHASHSEED'] = str(seed)
    
    # 尝试设置NumPy随机种子（如果可用）
    try:
        import numpy as np
        np.random.seed(seed)
        logger.debug(f"NumPy随机种子已设置为: {seed}")
    except ImportError:
        logger.debug("NumPy未安装，跳过NumPy随机种子设置")
    
    # 尝试设置PyTorch随机种子（如果可用）
    try:
        import torch
        torch.manual_seed(seed)
        if torch.cuda.is_available():
            torch.cuda.manual_seed(seed)
            torch.cuda.manual_seed_all(seed)
        
        if enable_deterministic:
            torch.backends.cudnn.deterministic = True
            torch.backends.cudnn.benchmark = False
        
        logger.debug(f"PyTorch随机种子已设置为: {seed}")
    except ImportError:
        logger.debug("PyTorch未安装，跳过PyTorch随机种子设置")
    
    logger.info(f"✅ 全局随机种子已设置为: {seed}")
    return seed


def load_seed_from_config(config_manager) -> int:
    """
    从配置管理器加载随机种子设置
    
    Args:
        config_manager: 配置管理器实例
        
    Returns:
        int: 随机种子值
    """
    try:
        reproducibility_config = config_manager.main_config.get('reproducibility', {})
        seed = reproducibility_config.get('random_seed', 42)
        enable_deterministic = reproducibility_config.get('enable_deterministic', True)
        
        return set_random_seed(seed, enable_deterministic)
    except Exception as e:
        logger.warning(f"从配置文件加载随机种子失败: {e}，使用默认种子42")
        return set_random_seed(42, True)


def get_random_state_info() -> dict:
    """
    获取当前随机状态信息，用于调试
    
    Returns:
        dict: 随机状态信息
    """
    info = {
        "python_random_state": random.getstate()[1][0],  # 获取当前状态的第一个值作为标识
        "pythonhashseed": os.environ.get('PYTHONHASHSEED', 'not_set')
    }
    
    try:
        import numpy as np
        info["numpy_random_state"] = np.random.get_state()[1][0]
    except ImportError:
        info["numpy_random_state"] = "not_available"
    
    try:
        import torch
        info["torch_random_state"] = torch.initial_seed()
        info["torch_deterministic"] = torch.backends.cudnn.deterministic
    except ImportError:
        info["torch_random_state"] = "not_available"
        info["torch_deterministic"] = "not_available"
    
    return info