import json
import logging
import time

from aide.backend.utils import (
    FunctionSpec,
    OutputType,
    opt_messages_to_list,
    backoff_create,
)
from funcy import notnone, once, select_values

import openai
from openai import OpenAI

logger = logging.getLogger("aide")

_client: OpenAI = None

OPENAI_TIMEOUT_EXCEPTIONS = (
    openai.RateLimitError,
    openai.APIConnectionError,
    openai.APITimeoutError,
    openai.InternalServerError,
    json.JSONDecodeError, 
    Exception,
)

@once
def _setup_openai_client():
    global _client
    _client = OpenAI(
        base_url="https://api.deepseek.com",
        api_key="sk-d711c85f45764811977cacd65c0c23ad",
        max_retries=500,
    )

def create_and_decode(*args, **kwargs):
    func_spec = kwargs.pop("func_spec", None)

    completion = _client.chat.completions.create(*args, **kwargs)
    if not completion or not hasattr(completion, "choices"):
        raise ValueError("APIから有効なcompletionが返りませんでした。")
    choice = completion.choices[0]

    if func_spec is None:
        output = choice.message.content
    else:
        assert choice.message.tool_calls, (
            f"function_call is empty, it is not a function call: {choice.message}"
        )
        assert (
            choice.message.tool_calls[0].function.name == func_spec.name
        ), "Function name mismatch"
        try:
            output = json.loads(choice.message.tool_calls[0].function.arguments)
        except json.JSONDecodeError as e:
            logger.error(
                f"Error decoding the function arguments: "
                f"{choice.message.tool_calls[0].function.arguments}"
            )
            raise e

    return completion, output

def query(
    system_message: str | None,
    user_message: str | None,
    func_spec: FunctionSpec | None = None,
    convert_system_to_user: bool = False,
    **model_kwargs,
) -> tuple[OutputType, float, int, int, dict]:
    _setup_openai_client()
    filtered_kwargs: dict = select_values(notnone, model_kwargs)  # type: ignore

    messages = opt_messages_to_list(
        system_message,
        user_message,
        convert_system_to_user=convert_system_to_user,
    )

    logger.info("---------------------------------------")
    for i, message in enumerate(messages):
        logger.info(f"Message {i}: {message}")
    logger.info("---------------------------------------")

    if func_spec is not None:
        filtered_kwargs["tools"] = [func_spec.as_openai_tool_dict]
        filtered_kwargs["tool_choice"] = func_spec.openai_tool_choice_dict

    t0 = time.time()
    result = backoff_create(
        create_and_decode,
        # _client.chat.completions.create,
        OPENAI_TIMEOUT_EXCEPTIONS,
        func_spec=func_spec,
        messages=messages,
        **filtered_kwargs,
    )
    req_time = time.time() - t0

    # choice = completion.choices[0]

    # if func_spec is None:
    #     output = choice.message.content
    # else:
    #     assert (
    #         choice.message.tool_calls
    #     ), f"function_call is empty, it is not a function call: {choice.message}"
    #     assert (
    #         choice.message.tool_calls[0].function.name == func_spec.name
    #     ), "Function name mismatch"
    #     try:
    #         output = json.loads(choice.message.tool_calls[0].function.arguments)
    #     except json.JSONDecodeError as e:
    #         logger.error(
    #             f"Error decoding the function arguments: {choice.message.tool_calls[0].function.arguments}"
    #         )
    #         raise e

    completion, output = result

    usage = getattr(completion, "usage", None)
    in_tokens = getattr(usage, "prompt_tokens", None)
    out_tokens = getattr(usage, "completion_tokens", None)

    info = {
        "system_fingerprint": getattr(completion, "system_fingerprint", None),
        "model": getattr(completion, "model", None),
        "created": getattr(completion, "created", None),
    }

    return output, req_time, in_tokens, out_tokens, info