from typing import Any, Dict, List, Optional

from llm_mcts.data_types import Action
from llm_mcts.llm_generation_interface import GenerationRequest, GenerationResult
from llm_mcts.mcts_algo.eval_result import EvalResult, EvalResultLiveCodeBench
from llm_mcts.prompt_configs import PromptConfig
from llm_mcts.prompts.base import PromptTemplate
from llm_mcts.tasks.live_code_bench_code_generation.problem import CodeGenerationProblem
from llm_mcts.tasks.live_code_bench_code_generation.task import (
    LiveCodeBenchCodeGenerationTask,
)


class LiveCodeBenchPromptV10(PromptTemplate):
    def __init__(
        self, prompt_config: PromptConfig, task: LiveCodeBenchCodeGenerationTask
    ):
        self.prompt_config = prompt_config
        self.task = task

    def initial_prompt(self) -> str:
        # based on the live code bench's prompt
        # https://github.com/LiveCodeBench/LiveCodeBench/blob/f05cda286956b0a976df08afe2e2a323358d32d1/lcb_runner/prompts/code_generation.py
        prompt = (
            PromptConstantsInitial.SYSTEM_MESSAGE_GENERIC
            + "\n\n"
            + get_generic_question_template_answer_initial(self.task.problem)
        )
        return prompt

    def feedback_prompt(
        self,
        action: Action,
        eval_results: Optional[List[EvalResult]],
        generation_result: GenerationResult,
    ) -> str:
        if eval_results is None:
            return "Your answer doesn't include any code."
        match action:
            case "answer":
                return answer_feedback_prompt(
                    problem=self.task.problem,
                    eval_results=eval_results,
                    code=generation_result.parse_python_code(),
                )
            case _:
                raise NotImplementedError(
                    f"feedback_prompt not implemented for action {action}"
                )

    def add_next_action_instruction(
        self, action: Action, next_prompt: GenerationRequest
    ) -> GenerationRequest:
        last_user_msg = next_prompt.messages[-1]
        assert last_user_msg.role == "user"

        # If the prompt.messages contains assistant message, it means that the prompt is not the first turn
        # len(prompt.messages) == 1 is not restricted to the first turn when the messages contain image prompts
        is_first_turn = True
        for msg in next_prompt.messages:
            if msg.role == "assistant":
                is_first_turn = False
                break
        return next_prompt


def answer_feedback_prompt(
    problem: CodeGenerationProblem,
    eval_results: List[EvalResultLiveCodeBench],
    code: str,
) -> str:
    """
    Refined feedback prompt to encourage a chain-of-thought approach during self-correction.
    We keep the function name unchanged and ensure that it returns a valid prompt
    ready for immediate use.
    """
    metadata = {}
    for i, eval_result in enumerate(eval_results):
        if eval_result.additional_info:
            metadata = eval_result.additional_info
            break

    # Build the final prompt string:
    # 1. SYSTEM_MESSAGE_GENERIC clarifies the context and expectations.
    # 2. get_generic_question_template_answer_self_repair() provides the question + user code + test outcomes.
    # 3. We add instructions to "think step by step" (chain of thought) before providing a final corrected solution.
    prompt = (
        PromptConstantsSelfRepair.SYSTEM_MESSAGE_GENERIC
        + "\n\n"
        + get_generic_question_template_answer_self_repair(problem, code, metadata)
        + "Now, let's analyze the code carefully and think step by step (in a chain of thought) about potential issues. "
        "Then provide a concise explanation (2-3 sentences) of what is wrong, followed by a fully corrected program. "
        "Enclose the entire fixed solution in code fences exactly once.\n\n"
    )

    return prompt


# ==========================
# Code Generation
# ==========================


class PromptConstantsInitial:
    SYSTEM_MESSAGE_GENERIC = f"You are an expert Python programmer. You will be given a question (problem specification) and will generate a correct Python program that matches the specification and passes all tests. You will NOT return anything except for the program."

    FORMATTING_MESSAGE_WITH_STARTER_CODE = "You will use the following starter code to write the solution to the problem and enclose your code within delimiters."

    FORMATTING_WITHOUT_STARTER_CODE = "Read the inputs from stdin solve the problem and write the answer to stdout (do not directly test on the sample inputs). Enclose your code within delimiters as follows."


def get_generic_question_template_answer_initial(question: CodeGenerationProblem):
    prompt = f"### Question:\n{question.question_content}\n\n"
    if question.starter_code:
        prompt += f"### Format: {PromptConstantsInitial.FORMATTING_MESSAGE_WITH_STARTER_CODE}\n"
        prompt += f"```python\n{question.starter_code}\n```\n\n"
    else:
        prompt += (
            f"### Format: {PromptConstantsInitial.FORMATTING_WITHOUT_STARTER_CODE}\n"
        )
        prompt += "```python\n# YOUR CODE HERE\n```\n\n"
    prompt += f"### Answer: (use the provided format with backticks)\n\n"
    return prompt


# ==========================
# Self-Repairs
# ==========================


class PromptConstantsSelfRepair:
    SYSTEM_MESSAGE_GENERIC = f"You are a helpful programming assistant and an expert Python programmer. You are helping a user write a program to solve a problem. The user has written some code, but it has some errors and is not passing the tests. You will help the user by first giving a concise (at most 2-3 sentences) textual explanation of what is wrong with the code. After you have pointed out what is wrong with the code, you will then generate a fixed version of the program. You must put the entired fixed program within code delimiters only for once."

    FORMATTING_WITHOUT_STARTER_CODE = "Read the inputs from stdin solve the problem and write the answer to stdout (do not directly test on the sample inputs). Enclose your code within delimiters as follows."


def get_check_prompt(metadata: Dict[str, Any]):
    if "error_code" not in metadata:
        return (
            "The output is correct. Please make it more efficient and robust to bugs."
        )
    if metadata["error_code"] == -1:
        # time limit exceeded
        message = f"The output is incorrect and got the following compilation error.\n{metadata['error']}"
    elif metadata["error_code"] == -2:
        # wrong answer
        message = f"The output is incorrect and got a wrong answer.\nInput: {metadata['inputs']}\nGenerated Output: {metadata['output']}\nExpected: {metadata['expected']}"
    elif metadata["error_code"] == -3:
        # time limit exceeded
        message = f"The output is incorrect and got time limit exceeded.\n{metadata['error']}\nInput: {metadata['inputs']}\nExpected: {metadata['expected']}"
    elif metadata["error_code"] == -4:
        # runtime error
        message = f"The output is incorrect and got a runtime error.\nInput: {metadata['inputs']}\nExpected: {metadata['expected']}\n{metadata['error']}"
    else:
        raise NotImplementedError(
            f"metadata['error_code'] = {metadata['error_code']} not implemented || {metadata=}"
        )
    return message


def get_generic_question_template_answer_self_repair(
    question: CodeGenerationProblem, code: str, metadata: List[Dict[str, Any]]
) -> str:
    prompt = ""
    # prompt += f"### Question:\n{question.question_content}\n\n"
    prompt += f"### Answer:\n```python\n{code}\n```\n\n"
    for i, single_metadata in enumerate(metadata):
        prompt += f"### Public test {i+1}:\n{get_check_prompt(single_metadata)}\n\n"
    prompt += (
        f"### Format: {PromptConstantsSelfRepair.FORMATTING_WITHOUT_STARTER_CODE}\n"
    )
    prompt += "```python\n# YOUR CODE HERE\n```\n\n"
    prompt += f"### Answer: (use the provided format with backticks)\n\n"
    return prompt
