from typing import List, Optional, Dict, Any

from llm_mcts.data_types import Action
from llm_mcts.llm_generation_interface import GenerationRequest, GenerationResult
from llm_mcts.mcts_algo.eval_result import EvalResult, EvalResultWithAns
from llm_mcts.prompt_configs import PromptConfig
from llm_mcts.prompts.base import PromptTemplate
from llm_mcts.tasks.code_contest.task import CodeContestTask
from llm_mcts.tasks.code_contest.problem import CodeContestProblem


class CodeContestLCBSingleTurnPrompt(PromptTemplate):
    def __init__(self, prompt_config: PromptConfig, task: CodeContestTask):
        self.task = task

    def initial_prompt(self) -> str:
        # based on the live code bench's prompt
        # https://github.com/LiveCodeBench/LiveCodeBench/blob/f05cda286956b0a976df08afe2e2a323358d32d1/lcb_runner/prompts/code_generation.py
        prompt = PromptConstantsInitial.SYSTEM_MESSAGE_GENERIC + "\n\n"
        prompt += f"### Question:\n{self.task.problem.description}\n\n"
        prompt += (
            f"### Format: {PromptConstantsInitial.FORMATTING_WITHOUT_STARTER_CODE}\n"
        )
        prompt += "```python\n# YOUR CODE HERE\n```\n\n"
        prompt += f"### Answer: (use the provided format with backticks)\n\n"
        return prompt

    def feedback_prompt(
        self,
        action: Action,
        eval_results: Optional[List[EvalResult]],
        generation_result: GenerationResult,
    ) -> str:
        try:
            code = generation_result.parse_python_code()
        except:
            code = ""
        match action:
            case "answer":
                return answer_feedback_prompt(
                    problem=self.task.problem,
                    eval_results=eval_results,
                    code=code,
                )
            case _:
                raise NotImplementedError(
                    f"feedback_prompt not implemented for action {action}"
                )

    def add_next_action_instruction(
        self, action: Action, next_prompt: GenerationRequest
    ) -> GenerationRequest:
        last_user_msg = next_prompt.messages[-1]
        assert last_user_msg.role == "user"

        next_prompt.messages = next_prompt.messages[-1:]

        return next_prompt


def answer_feedback_prompt(
    problem: CodeContestProblem, eval_results: List[EvalResultWithAns], code: str
) -> str:

    prompt = PromptConstantsSelfRepair.SYSTEM_MESSAGE_GENERIC + "\n\n"
    prompt += f"### Question:\n{problem.description}\n\n"
    if code == "" or eval_results is None:
        prompt += f"### Answer: Your answer doesn't include any code.\n\n"
        prompt += (
            f"### Format: {PromptConstantsInitial.FORMATTING_WITHOUT_STARTER_CODE}\n"
        )
        prompt += "```python\n# YOUR CODE HERE\n```\n\n"
        prompt += f"### Answer: (use the provided format with backticks)\n\n"
        return prompt

    prompt += f"### Answer:\n```python\n{code}\n```\n\n"

    for i, eval_result in enumerate(eval_results):
        is_correct = eval_result.get_score() == 1
        output = eval_result.answer

        prompt += f"### Public test {i}\n\n"

        if is_correct:
            prompt += f"""
Input:
{problem.public_tests["input"][i]}
Expected Output:
{problem.public_tests["output"][i]}
Your Output:
{output}

Result: **Correct**

"""
        else:
            prompt += f"""
Input:
{problem.public_tests["input"][i]}
Expected Output:
{problem.public_tests["output"][i]}
Your Output:
{output}

Result: **Wrong**

"""

    prompt += (
        f"### Format: {PromptConstantsSelfRepair.FORMATTING_WITHOUT_STARTER_CODE}\n"
    )
    prompt += "```python\n# YOUR CODE HERE\n```\n\n"
    prompt += f"### Answer: (use the provided format with backticks)\n\n"

    prompt += (
        "Now, let's analyze the code carefully and think step by step (in a chain of thought) about potential issues. "
        "Then provide a concise explanation (2-3 sentences) of what is wrong, followed by a fully corrected program. "
        "Enclose the entire fixed solution in code fences exactly once.\n\n"
    )
    return prompt


# ==========================
# Code Generation
# ==========================


class PromptConstantsInitial:
    SYSTEM_MESSAGE_GENERIC = f"You are an expert Python programmer. You will be given a question (problem specification) and will generate a correct Python program that matches the specification and passes all tests. You will NOT return anything except for the program."

    FORMATTING_MESSAGE_WITH_STARTER_CODE = "You will use the following starter code to write the solution to the problem and enclose your code within delimiters."

    FORMATTING_WITHOUT_STARTER_CODE = "Read the inputs from stdin solve the problem and write the answer to stdout (do not directly test on the sample inputs). Enclose your code within delimiters as follows."


# ==========================
# Self-Repairs
# ==========================


class PromptConstantsSelfRepair:
    SYSTEM_MESSAGE_GENERIC = f"You are a helpful programming assistant and an expert Python programmer. You are helping a user write a program to solve a problem. The user has written some code, but it has some errors and is not passing the tests. You will help the user by first giving a concise (at most 2-3 sentences) textual explanation of what is wrong with the code. After you have pointed out what is wrong with the code, you will then generate a fixed version of the program. You must put the entired fixed program within code delimiters only for once."

    FORMATTING_WITHOUT_STARTER_CODE = "Read the inputs from stdin solve the problem and write the answer to stdout (do not directly test on the sample inputs). Enclose your code within delimiters as follows."
