from typing import List, Optional

from llm_mcts.data_types import Action
from llm_mcts.llm_generation_interface import GenerationRequest, GenerationResult
from llm_mcts.mcts_algo.eval_result import EvalResult, EvalResultWithScore
from llm_mcts.prompt_configs import PromptConfig
from llm_mcts.prompts.base import PromptTemplate
from llm_mcts.tasks.omni_math.task import OmniMathTask


class KouV1Prompt(PromptTemplate):
    def __init__(self, prompt_config: PromptConfig, task: OmniMathTask):
        self.task = task

    def initial_prompt(self) -> str:
        prompt = initial_prompt()
        prompt += problem_prompt(self.task.problem.problem)
        return prompt

    def feedback_prompt(
        self,
        action: Action,
        eval_results: Optional[List[EvalResult]],
        generation_result: GenerationResult,
    ) -> str:
        match action:
            case "answer":
                return answer_feedback_prompt(
                    eval_results=eval_results,
                )
            case _:
                raise NotImplementedError(
                    f"feedback_prompt not implemented for action {action}"
                )

    def add_next_action_instruction(
        self, action: Action, next_prompt: GenerationRequest
    ) -> GenerationRequest:
        last_user_msg = next_prompt.messages[-1]
        assert last_user_msg.role == "user"

        # If the prompt.messages contains assistant message, it means that the prompt is not the first turn
        # len(prompt.messages) == 1 is not restricted to the first turn when the messages contain image prompts
        is_first_turn = True
        for msg in next_prompt.messages:
            if msg.role == "assistant":
                is_first_turn = False
                break

        last_user_msg.content += next_task_prompt(action, is_first_turn=is_first_turn)
        return next_prompt


def problem_prompt(problem: str) -> str:
    return f"""
Problem:
{problem}
"""


def initial_prompt() -> str:
    task_explanation = """
You are an assistant who is very familiar with mathematics, and great at solving mathematical problem by detailed and careful reasoning.
You will be given a mathematical problem, so output your thinking and reasoning in solving the given problem. After that, clearly state the answer
to the given problem, either in English or TeX format. Please reason step by step, and put your final answer within \\boxed{}.
"""

    return task_explanation


def answer_feedback_prompt(eval_results: List[EvalResultWithScore]) -> str:
    assert len(eval_results) == 1

    prompt = f"""
Another AI assistant carefully evaluated your solution and gave a score to your solution by an integer from 0 to 5:
Score:
{eval_results[0].score}
Reason:
{eval_results[0].reason}
"""
    return prompt


def next_task_prompt(kind: Action, is_first_turn: bool) -> str:
    if kind != "answer":
        raise NotImplementedError()
    if is_first_turn:
        return ""
    else:
        return """
Given the above feedback from another AI assistant, carefully revise your solution by revisiting your reasoning and thinking to answer the given mathematical question. Put your final answer within \\boxed{}.
"""
