import argparse
from pathlib import Path

from llm_mcts.mcts_algo.mcts_result import MCTSResult
from llm_mcts.mcts_scorer.default import DefaultScorer
from llm_mcts.models.model_builder import build_model
from llm_mcts.tasks.omni_math.task import OmniMathTask
from llm_mcts.visualize_mcts import render_mcts_graph
from llm_mcts.tasks.omni_math.omni_judge import MODEL_NAME as OMNI_JUDGE_MODEL_NAME

logging_dir = (Path(__file__) / ".." / ".." / ".." / "logging" / "omni_math").resolve()


def parse_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser()
    parser.add_argument("-e", "--experiment_name", type=str, required=True)
    parser.add_argument("-i", "--idx", type=int, required=True)
    parser.add_argument("--judge_model", type=str, default="gpt-4o-2024-08-06")
    parser.add_argument("--judge_temperature", type=str, default="0.0")
    parser.add_argument("--reward_model_name", type=str, default=None)
    parser.add_argument("--only_reward_model", action="store_true")
    parser.add_argument("--is_sigmoid", action="store_true")
    parser.add_argument("--dataset_name", type=str)
    return parser.parse_args()


def main() -> None:
    args = parse_args()

    save_dir = logging_dir / args.experiment_name
    assert save_dir.is_dir(), f"Experiment {args.experiment_name} does not exist"

    mcts_result = MCTSResult.load(save_dir / f"mcts_result_{args.idx}.pkl")
    scorer = DefaultScorer()
    if args.judge_model != OMNI_JUDGE_MODEL_NAME:
        judge_model = build_model(
            model_names=args.judge_model,
            model_probs=None,
            temperatures=args.judge_temperature,
            logging_dir=save_dir / f"test_{args.idx}",
        )
    else:
        judge_model = args.judge_model

    task = OmniMathTask.load_record(
        idx=args.idx,
        judge_model=judge_model,
        dataset_name=args.dataset_name,
        reward_model_name=args.reward_model_name,
        only_reward_model=args.only_reward_model,
        is_sigmoid=args.is_sigmoid,
    )
    render_mcts_graph(
        mcts_result, scorer, task, save_dir / f"mcts_graph_{args.idx}", view=False
    )


if __name__ == "__main__":
    main()
