#!/bin/bash
set -e

# ==============================================================================
# Configuration
# ==============================================================================
ANSWER_MODELS=$1
NUM_SIM_ARG=$2
NUM_SIMULATIONS=$((NUM_SIM_ARG - 1))
NUM_EXPAND_SAMPLES=$3
INITIAL_EXPAND_SAMPLES=$NUM_EXPAND_SAMPLES

PROMPT_TYPE="baseline_v1_single_turn"

# Model configuration
TEMPERATURE=$4

MCTS_ALGO="standard"
NUM_TRIAL=$5

# Parallel processing configuration
N_JOBS=16  # Number of parallel jobs

EXPERIMENT_NAME="${MCTS_ALGO}_${ANSWER_MODELS}_temp${TEMPERATURE}_nsim${NUM_SIMULATIONS}_trial${NUM_TRIAL}"
EXPERIMENT_NAME="${EXPERIMENT_NAME//\//_}"


# Base directory setup
BASE_DIR="logging/arc/${EXPERIMENT_NAME}"
mkdir -p "${BASE_DIR}/logs"

# Load problem IDs from file
PROBLEMS_FILE="llm_mcts/tasks/arc/greenblat_blog_eval_100_probs.txt"
TRAINING_IDS=($(cat "${PROBLEMS_FILE}" | sed 's/\.json//g'))

# ==============================================================================
# Solve problems
# ==============================================================================
process_problem() {
    local id=$1
    local mcts_path="${BASE_DIR}/mcts_result_${id}.pkl"

    # Check if the pickle file already exists
    if [ -f "$mcts_path" ]; then
        echo "Skipping problem ${id} - pickle file already exists"
    else
        echo "Processing problem ${id}"

        # Solve problem
        python scripts/arc/run_arc.py \
            --experiment_name "${EXPERIMENT_NAME}" \
            --idx "${id}" \
            --split "evaluation" \
            --temperature "${TEMPERATURE}" \
            --answer_models "${ANSWER_MODELS}" \
            --initial_prompt_type "${PROMPT_TYPE}" \
            --num_simulations $NUM_SIMULATIONS \
            --num_expand_samples $NUM_EXPAND_SAMPLES \
            --initial_expand_samples $INITIAL_EXPAND_SAMPLES \
            --mcts_algo "${MCTS_ALGO}" \
            2>&1 | tee "${BASE_DIR}/logs/${id}.log"
    fi

    # Check if pickle file exists now (either pre-existing or newly created)
    if [ -f "$mcts_path" ]; then
        # Create submission directory if it doesn't exist
        mkdir -p "${BASE_DIR}/prediction_${id}"

        # Submit solution
        python scripts/arc/make_submission_arc.py \
            --mcts-path="${mcts_path}" \
            --result-path="${BASE_DIR}/prediction_${id}/result.txt" \
            --problem_path="ARC-AGI/data/evaluation/${id}.json" \
            2>&1 | tee "${BASE_DIR}/logs/${id}_submission.log"
    else
        echo "Warning: Cannot submit solution for ${id} - pickle file does not exist"
    fi
}
export -f process_problem
export BASE_DIR
export EXPERIMENT_NAME
export TEMPERATURE
export ANSWER_MODELS
export MCTS_ALGO
export NUM_SIMULATIONS
export PROMPT_TYPE
export NUM_EXPAND_SAMPLES
export INITIAL_EXPAND_SAMPLES

# ==============================================================================
# Main execution
# ==============================================================================
start_time=$(date +%s)

# Process problems in parallel
printf "%s\n" "${TRAINING_IDS[@]}" | parallel --line-buffer -j ${N_JOBS} process_problem

end_time=$(date +%s)
elapsed_time=$((end_time - start_time))
elapsed_time=$(echo "scale=2; $elapsed_time / 60" | bc)
echo "Elapsed time (solve + submit): $elapsed_time minutes"

# ==============================================================================
# Gather results and calculate scores
# ==============================================================================
python scripts/arc/gather_results_arc.py \
    --log-dir "${BASE_DIR}" \
    2>&1 | tee "${BASE_DIR}/logs/gather_results.log"

end_time=$(date +%s)
elapsed_time=$((end_time - start_time))
elapsed_time=$(echo "scale=2; $elapsed_time / 60" | bc)
echo "Total elapsed time: $elapsed_time minutes"
