import os
from datetime import datetime
from pathlib import Path
from typing import Tuple


def get_logging_dir(id: str | int) -> Path:
    logging_dir = (
        Path(__file__).parent.parent
        / "logging"
        / f"{datetime.now().strftime('%Y_%m_%d_%H%M%S')}_{id}"
    ).resolve()
    logging_dir.mkdir(parents=True, exist_ok=True)

    return logging_dir


def get_problem_info(
    problem_name: str,
    dataset: str | None = None,
    llm_mcts_path: str | None = None,
) -> Tuple[Path | str | None, str]:
    """
    Get the problem path and problem name from the problem name.
    If the problem name is not a path, it is assumed to be a problem name.
    If the problem name is a path, it is assumed to be a problem path.
    """
    if dataset is None:
        problem_basename = problem_name.split("/")[-1]
        # Infer dataset by checking if path exists
        if (
            problem_path_cand := Path(
                f"{llm_mcts_path}/ARC-AGI/data/training/{problem_basename}"
            )
        ).exists():
            problem_path = problem_path_cand
        elif (
            problem_path_cand := Path(
                f"{llm_mcts_path}/ARC-AGI/data/evaluation/{problem_basename}"
            )
        ).exists():
            problem_path = problem_path_cand
        elif (
            problem_path_cand := Path(
                f"{llm_mcts_path}/gpt4_blog_train_rearc/{problem_basename.removeprefix('rearc_')}"
            )
        ).exists():
            problem_path = problem_path_cand
        else:
            problem_path = None
        return problem_path, problem_basename

    # problem_path is the path to the problem json file
    if os.path.exists(problem_name):
        problem_path = problem_name
    else:
        problem_path = f"{llm_mcts_path}/ARC-AGI/data/{dataset}/{problem_name}"

    # problem_name is an unique task id (training set has 400 tasks, evaluation set has 400 tasks)
    problem_name = problem_path.split("/")[-1]

    return problem_path, problem_name
