from dataclasses import dataclass
from typing import List, Literal, Optional, TypeAlias

from PIL.Image import Image


MathVistaSplit: TypeAlias = Literal["testmini", "test"]
MathVistaLanguage: TypeAlias = Literal["English", "Chinese", "Persian"]
MathVistaCategory: TypeAlias = Literal["math-targeted-vqa", "general-vqa"]


@dataclass
class MathVistaMetaData:
    split: MathVistaSplit  # Data split: "testmini" or "test"
    language: MathVistaLanguage  # Question language: "English", "Chinese", or "Persian"
    img_width: int  # The width of the associated image in pixels
    img_height: int  # The height of the associated image in pixels
    source: str  # The source dataset from which the problem was taken
    category: MathVistaCategory  # The category of the problem: "math-targeted-vqa" or "general-vqa"
    task: str  # The task of the problem, e.g., "geometry problem solving"
    context: str  # The visual context type of the associated image
    grade: str  # The grade level of the problem, e.g., "high school"
    skills: List[str]  # A list of mathematical reasoning skills that the problem tests


MathVistaQuestionType: TypeAlias = Literal["multi_choice", "free_form"]
MathVistaAnswerType: TypeAlias = Literal["text", "integer", "float", "list"]


@dataclass
class MathVistaProblem:
    pid: str  # Problem ID
    question: str  # The question text
    query: str  # The query text used as input (prompt) for the evaluation model
    image: str  # A file path pointing to the associated image
    decoded_image: Image  # Decoded image
    choices: Optional[
        List[str]
    ]  # Choice options for multiple-choice problems. For free-form problems, this could be a 'none' value
    unit: Optional[
        str
    ]  # The unit associated with the answer, e.g., "m^2", "years". If no unit is relevant, it can be a 'none' value
    precision: Optional[
        int
    ]  # The number of decimal places the answer should be rounded to
    question_type: (
        MathVistaQuestionType  # The type of question: "multi_choice" or "free_form"
    )
    answer_type: MathVistaAnswerType  # The format of the answer: "text", "integer", "float", or "list"
    metadata: MathVistaMetaData  # Metadata (see MathVistaMetaData)
    answer: str  # The correct answer for the problem
    caption: str = ""  # TODO: add support for caption
    ocr: str = ""  # TODO: add support for OCR
