import argparse
import json
import re
from pathlib import Path
from typing import List

from llm_mcts.mcts_algo.mcts_result import MCTSResult
from llm_mcts.tasks.swe_bench.task import SWEBenchTask
from llm_mcts.node_ranker.swe_bench_simple_ranker import SWEBenchSimpleRanker

logging_dir = (Path(__file__) / ".." / ".." / ".." / "logging" / "swe_bench").resolve()


def extract_patch(text: str) -> str:
    """
    Extracts a `patch` block from the given text. If no patch block is found, returns an empty string.
    """
    if "```patch" in text:
        pattern = r"```patch(.*?)```"
    elif "<patch>" in text:
        pattern = r"<patch>(.*?)</patch>"
    else:
        return ""
    match = re.search(pattern, text, re.DOTALL)
    return match.group(1).strip() if match else ""


def parse_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser()
    parser.add_argument("-e", "--experiment_name", type=str, required=True)
    parser.add_argument("-i", "--idx", type=int, required=True)
    parser.add_argument(
        "--dataset_name", type=str, default="princeton-nlp/SWE-bench_Lite_bm25_13K"
    )
    parser.add_argument("--split", type=str, default="dev")
    return parser.parse_args()


def main() -> None:
    args = parse_args()

    save_dir = logging_dir / args.experiment_name
    assert save_dir.is_dir(), f"Experiment {args.experiment_name} does not exist"

    save_result_json_dir = save_dir / "each_result"
    save_result_json_dir.mkdir(parents=True, exist_ok=True)

    # Load task
    task = SWEBenchTask.load_record(
        idx=args.idx, dataset_name=args.dataset_name, split=args.split
    )
    instance_id = task.problem.instance_id

    # Load mcts result
    mcts_path = save_dir / f"mcts_result_{args.idx}.pkl"
    assert mcts_path.exists()
    mcts_result = MCTSResult.load(mcts_path)

    # Get top submission
    top_node_list = SWEBenchSimpleRanker().top_k_predictions(
        mcts_result,
    )
    for node in top_node_list:
        generated_completion = node.completion
        if generated_completion is None:
            continue
        result_patch = extract_patch(generated_completion.generation)
        if result_patch != "":
            break

    with open(save_result_json_dir / f"{instance_id}.json", "w") as f:
        json.dump(
            {
                "instance_id": instance_id,
                "model_name_or_path": args.experiment_name,
                "model_patch": result_patch,
            },
            f,
        )

    with open(save_result_json_dir / f"{instance_id}_llm_judge_score.json", "w") as f:
        json.dump(
            {
                "instance_id": instance_id,
                "llm_judge_score": (
                    node.eval_results[0].score if node.eval_results is not None else 0
                ),
            },
            f,
        )

    print(f"Saved to {save_result_json_dir / f'{instance_id}.json'}")


if __name__ == "__main__":
    main()
