import torch
import torch.nn as nn
from torch.nn import functional as F
import numpy as np
from sklearn.model_selection import train_test_split
import csv
import os
import torch.optim as optim

# hyperparameters
batch_size = 64 # how many independent sequences will we process in parallel?
block_size = 1024 # what is the maximum context length for predictions?
max_iters = 10000
eval_interval = 50
learning_rate = 3e-4
device = 'cuda:0' if torch.cuda.is_available() else 'cpu'
print(device)
eval_iters = 1000
n_embd = 384
n_head = 6
n_layer = 6
dropout = 0.2
num_labels = 2
# ------------

filename = 'Train_Align_Val_Align_Aug_temp_withlabel'
Trainingdata = 'SequenceTotalTrainingAligntempwithlabel.npy'
Validationdata = 'SequenceTotalValidationAligntempwithlabel.npy'

torch.manual_seed(1337)

# Define the device base names
nm_np_bases = ["{}_D", "{}_G", "{}_S", "{}_B"]
npn_pnp_bases = ["{}_C", "{}_B", "{}_E"]
c_r_l_i_bases = ["{}_P", "{}_N"]
xor_bases = ["{}_A", "{}_B", "{}_VDD", "{}_VSS", "{}_Y"]
pfd_bases = ["{}_A", "{}_B", "{}_QA", "{}_QB", "{}_VDD", "{}_VSS"]
inverter_bases = ["{}_A", "{}_Q", "{}_VDD", "{}_VSS"]
transmission_gate_bases = ["{}_A", "{}_B", "{}_C", "{}_VDD", "{}_VSS"]

# Initialize the list of NM, PM, C, R, L, I, VIN, VB, VOUT devices, and additional entries
devices = []
for prefix in ["NM", "PM"]:
    for i in range(1, 26):
        devices.append(f"{prefix}{i}")
        for base in nm_np_bases:
            devices.append(base.format(f"{prefix}{i}"))

for prefix in ["NPN", "PNP"]:
    for i in range(1, 26):
        devices.append(f"{prefix}{i}")
        for base in npn_pnp_bases:
            devices.append(base.format(f"{prefix}{i}"))

for i in range(1, 26):
    devices.append(f"R{i}")
    for base in c_r_l_i_bases:
        devices.append(base.format(f"R{i}"))

for i in range(1, 26):
    devices.append(f"C{i}")
    for base in c_r_l_i_bases:
        devices.append(base.format(f"C{i}"))

for i in range(1, 26):
    devices.append(f"L{i}")
    for base in c_r_l_i_bases:
        devices.append(base.format(f"L{i}"))

for i in range(1, 26):
    devices.append(f"DIO{i}")
    for base in c_r_l_i_bases:
        devices.append(base.format(f"DIO{i}"))

for i in range(1, 6):
    devices.append(f"XOR{i}")
    for base in xor_bases:
        devices.append(base.format(f"XOR{i}"))

for i in range(1, 6):
    devices.append(f"PFD{i}")
    for base in pfd_bases:
        devices.append(base.format(f"PFD{i}"))

for i in range(1, 11):
    devices.append(f"INVERTER{i}")
    for base in inverter_bases:
        devices.append(base.format(f"INVERTER{i}"))

for i in range(1, 11):
    devices.append(f"TRANSMISSION_GATE{i}")
    for base in transmission_gate_bases:
        devices.append(base.format(f"TRANSMISSION_GATE{i}"))

# for i in range(1, 26):
#     devices.append(f"I{i}")
#     for base in c_r_l_i_bases:
#         devices.append(base.format(f"I{i}"))

for i in range(1, 6):
    devices.append(f"VIN{i}")

for i in range(1, 6):
    devices.append(f"IIN{i}")

for i in range(1, 6):
    devices.append(f"VOUT{i}")

for i in range(1, 6):
    devices.append(f"IOUT{i}")

for i in range(1, 11):
    devices.append(f"VB{i}")

for i in range(1, 11):
    devices.append(f"IB{i}")

for i in range(1, 6):
    devices.append(f"VCONT{i}")

for i in range(1, 6):
    devices.append(f"VCLK{i}")

for i in range(1, 6):
    devices.append(f"VCM{i}")

for i in range(1, 6):
    devices.append(f"VREF{i}")

for i in range(1, 6):
    devices.append(f"IREF{i}")

for i in range(1, 6):
    devices.append(f"VRF{i}")

for i in range(1, 6):
    devices.append(f"VLO{i}")

for i in range(1, 6):
    devices.append(f"VIF{i}")

for i in range(1, 6):
    devices.append(f"VBB{i}")

for i in range(1, 3):
    devices.append(f"LOGICA{i}")

for i in range(1, 3):
    devices.append(f"LOGICB{i}")

for i in range(1, 3):
    devices.append(f"LOGICD{i}")

for i in range(1, 3):
    devices.append(f"LOGICF{i}")

for i in range(1, 3):
    devices.append(f"LOGICG{i}")

for i in range(1, 3):
    devices.append(f"LOGICQ{i}")

for i in range(1, 3):
    devices.append(f"LOGICQA{i}")

for i in range(1, 3):
    devices.append(f"LOGICQB{i}")

# Adding the additional entries
additional_entries = ["VDD", "VSS", "TRUNCATE"]
devices.extend(additional_entries)

# Create a mapping from device names to integers and vice versa
stoi = { device: i for i, device in enumerate(devices) }
itos = { i:device for i,device in enumerate(devices) }
vocab_size = len(devices)

# Print the results
# print("Devices in order:", devices)
# print("Vocabulary size:", len(devices))
# print("Device to index mapping:", stoi)
# print("Index to device mapping:", itos)

encode = lambda s: [stoi[c] for c in s]
decode = lambda l: '->'.join([itos[i] for i in l]) + '->'

# Load the training data
train_data = np.load(Trainingdata)

# Strip out the last column (which is a string) and convert it to an integer
# Adjust labels to start from 0 by subtracting 1
train_data_last_col = train_data[:, -1].astype(int) - 1
train_data_main = train_data[:, :-1]

# Encode the main data (without the last column)
encoded_train_data = encode(train_data_main.flatten())

# Convert to tensor and reshape to original shape without the last column
encoded_train_tensor = torch.tensor(encoded_train_data, dtype=torch.long).view(train_data_main.shape)

# Add back the last column to the tensor, converting it to a tensor first
train_data = torch.cat((encoded_train_tensor, torch.tensor(train_data_last_col, dtype=torch.long).view(-1, 1)), dim=1)

# Load the validation data
val_data = np.load(Validationdata)

# Strip out the last column (which is a string) and convert it to an integer
# Adjust labels to start from 0 by subtracting 1
val_data_last_col = val_data[:, -1].astype(int) - 1
val_data_main = val_data[:, :-1]

# Encode the main data (without the last column)
encoded_val_data = encode(val_data_main.flatten())

# Convert to tensor and reshape to original shape without the last column
encoded_val_tensor = torch.tensor(encoded_val_data, dtype=torch.long).view(val_data_main.shape)

# Add back the last column to the tensor, converting it to a tensor first
val_data = torch.cat((encoded_val_tensor, torch.tensor(val_data_last_col, dtype=torch.long).view(-1, 1)), dim=1)

# Print shapes to confirm
print(train_data.shape)
print(val_data.shape)

class Head(nn.Module):
    """ one head of self-attention """

    def __init__(self, head_size):
        super().__init__()
        self.key = nn.Linear(n_embd, head_size, bias=False)
        self.query = nn.Linear(n_embd, head_size, bias=False)
        self.value = nn.Linear(n_embd, head_size, bias=False)
        self.register_buffer('tril', torch.tril(torch.ones(block_size, block_size)))

        self.dropout = nn.Dropout(dropout)

    def forward(self, x):
        # input of size (batch, time-step, channels)
        # output of size (batch, time-step, head size)
        B,T,C = x.shape
        k = self.key(x)   # (B,T,hs)
        q = self.query(x) # (B,T,hs)
        # compute attention scores ("affinities")
        wei = q @ k.transpose(-2,-1) * k.shape[-1]**-0.5 # (B, T, hs) @ (B, hs, T) -> (B, T, T)
        wei = wei.masked_fill(self.tril[:T, :T] == 0, float('-inf')) # (B, T, T)
        wei = F.softmax(wei, dim=-1) # (B, T, T)
        wei = self.dropout(wei)
        # perform the weighted aggregation of the values
        v = self.value(x) # (B,T,hs)
        out = wei @ v # (B, T, T) @ (B, T, hs) -> (B, T, hs)
        return out

class MultiHeadAttention(nn.Module):
    """ multiple heads of self-attention in parallel """

    def __init__(self, num_heads, head_size):
        super().__init__()
        self.heads = nn.ModuleList([Head(head_size) for _ in range(num_heads)])
        self.proj = nn.Linear(head_size * num_heads, n_embd)
        self.dropout = nn.Dropout(dropout)

    def forward(self, x):
        out = torch.cat([h(x) for h in self.heads], dim=-1)
        out = self.dropout(self.proj(out))
        return out

class FeedFoward(nn.Module):
    """ a simple linear layer followed by a non-linearity """

    def __init__(self, n_embd):
        super().__init__()
        self.net = nn.Sequential(
            nn.Linear(n_embd, 4 * n_embd),
            nn.ReLU(),
            nn.Linear(4 * n_embd, n_embd),
            nn.Dropout(dropout),
        )

    def forward(self, x):
        return self.net(x)

class Block(nn.Module):
    """ Transformer block: communication followed by computation """

    def __init__(self, n_embd, n_head):
        # n_embd: embedding dimension, n_head: the number of heads we'd like
        super().__init__()
        head_size = n_embd // n_head
        self.sa = MultiHeadAttention(n_head, head_size)
        self.ffwd = FeedFoward(n_embd)
        self.ln1 = nn.LayerNorm(n_embd)
        self.ln2 = nn.LayerNorm(n_embd)

    def forward(self, x):
        x = x + self.sa(self.ln1(x))
        x = x + self.ffwd(self.ln2(x))
        return x

class RewardModel(nn.Module):
    def __init__(self, vocab_size, n_embd, n_head, n_layer, num_labels, block_size, dropout):
        super().__init__()
        self.num_labels = num_labels
        
        # Token embedding and position embedding layers
        self.token_embedding_table = nn.Embedding(vocab_size, n_embd)
        self.position_embedding_table = nn.Embedding(block_size, n_embd)
        self.blocks = nn.Sequential(*[Block(n_embd, n_head=n_head) for _ in range(n_layer)])
        self.ln_f = nn.LayerNorm(n_embd)  # Final layer norm
        self.classifier_head = nn.Linear(n_embd, num_labels)  # Multi-class classification

        # Initialize weights
        self.apply(self._init_weights)

    def _init_weights(self, module):
        if isinstance(module, nn.Linear):
            torch.nn.init.normal_(module.weight, mean=0.0, std=0.02)
            if module.bias is not None:
                torch.nn.init.zeros_(module.bias)
        elif isinstance(module, nn.Embedding):
            torch.nn.init.normal_(module.weight, mean=0.0, std=0.02)

    def forward(self, input_ids):
        B, T = input_ids.shape

        # idx is a (B,T) tensor of integers
        tok_emb = self.token_embedding_table(input_ids)  # (B,T,C)
        pos_emb = self.position_embedding_table(torch.arange(T, device=input_ids.device))  # (T,C)
        x = tok_emb + pos_emb  # (B,T,C)
        x = self.blocks(x)  # (B,T,C)
        x = self.ln_f(x)  # (B,T,C)

        # Use the hidden state of the last token in the sequence
        logits = self.classifier_head(x[:, -1, :])  # (B, num_labels)
        return logits

# Calculate class weights based on the data
def calculate_class_weights(data):
    labels = data[:, -1]
    class_counts = torch.bincount(labels.long())
    print(class_counts)
    class_weights = 1.0 / (class_counts.float() + 1e-5)  # Add small value to avoid division by zero
    class_weights = class_weights / class_weights.sum() * len(class_counts)  # Normalize to the number of classes
    return class_weights

# Adjust get_batch to provide sequences and labels
def get_batch(data, batch_size, device):
    ix = torch.randint(len(data), (batch_size,))
    
    x = torch.stack([data[i, :-2] for i in ix])
    labels = torch.stack([data[i, -1] for i in ix]).long()
    
    x, labels = x.to(device), labels.to(device)
    
    return x, labels

# Function to calculate accuracy
def calculate_accuracy(logits, labels):
    preds = torch.argmax(logits, dim=1)
    correct = (preds == labels).float().sum()
    accuracy = correct / labels.size(0)
    return accuracy.item()

# Initialize the model
model = RewardModel(vocab_size, n_embd, n_head, n_layer, num_labels, block_size, dropout)
model.to(device)

# Create an optimizer
optimizer = optim.AdamW(model.parameters(), lr=learning_rate)

# Calculate class weights for training and validation sets
train_class_weights = calculate_class_weights(train_data).to(device)
val_class_weights = calculate_class_weights(val_data).to(device)
print("Train class weight: ", train_class_weights)
print("Val class weight: ", val_class_weights)

# Define loss functions for training and validation sets
train_loss_fn = nn.CrossEntropyLoss(weight=train_class_weights)
val_loss_fn = nn.CrossEntropyLoss(weight=val_class_weights)

# Initialize a variable to keep track of the best validation loss
best_val_loss = float('inf')

# Open a CSV file to write the losses and accuracy
with open(filename + '.csv', mode='w', newline='') as csv_file:
    csv_writer = csv.writer(csv_file)
    csv_writer.writerow(['Step', 'Train Loss', 'Validation Loss', 'Train Accuracy', 'Validation Accuracy'])

    # Define the loss and accuracy estimation function
    @torch.no_grad()
    def estimate_loss_and_accuracy():
        out = {}
        model.eval()
        for split in ['train', 'val']:
            losses = torch.zeros(eval_iters)
            accuracies = torch.zeros(eval_iters)
            for k in range(eval_iters):
                X, Y = get_batch(train_data if split == 'train' else val_data, batch_size, device)
                logits = model(X)
                loss = train_loss_fn(logits, Y) if split == 'train' else val_loss_fn(logits, Y)
                accuracy = calculate_accuracy(logits, Y)
                losses[k] = loss.item()
                accuracies[k] = accuracy
            out[split] = {
                'loss': losses.mean(),
                'accuracy': accuracies.mean()
            }
        model.train()
        return out

    # Training loop
    for iter in range(max_iters):
        torch.cuda.empty_cache()

        # Evaluate the loss and accuracy on the training and validation sets at intervals
        if iter % eval_interval == 0 or iter == max_iters - 1:
            metrics = estimate_loss_and_accuracy()
            print(f"step {iter}: train loss {metrics['train']['loss']:.4f}, val loss {metrics['val']['loss']:.4f}, "
                  f"train accuracy {metrics['train']['accuracy']:.4f}, val accuracy {metrics['val']['accuracy']:.4f}")
            csv_writer.writerow([iter, metrics['train']['loss'].item(), metrics['val']['loss'].item(),
                                 metrics['train']['accuracy'].item(), metrics['val']['accuracy'].item()])

            # Save the model if it has the best validation loss so far
            if metrics['val']['loss'] < best_val_loss:
                best_val_loss = metrics['val']['loss']
                torch.save(model.state_dict(), filename + '.pth')

        # Sample a batch of data
        xb, yb = get_batch(train_data, batch_size, device)

        # Forward pass to calculate loss
        logits = model(xb)
        loss = train_loss_fn(logits, yb)

        # Backward pass and optimization
        optimizer.zero_grad(set_to_none=True)
        loss.backward()
        optimizer.step()