import torch
import torch.nn as nn
from torch.nn import functional as F
import numpy as np
from sklearn.model_selection import train_test_split
import csv
import os

# hyperparameters
batch_size = 64 # how many independent sequences will we process in parallel?
block_size = 1024 # what is the maximum context length for predictions?
max_iters = 50000
eval_interval = 500
learning_rate = 3e-4
device = 'cuda:0' if torch.cuda.is_available() else 'cpu'
print(device)
eval_iters = 1000
n_embd = 384
n_head = 6
n_layer = 6
dropout = 0.2
# ------------

filename = 'check_param_size'
Trainingdata = 'SequenceTotalTrainingAligntemp.npy'
Validationdata = 'SequenceTotalValidationAligntemp.npy'

torch.manual_seed(1337)

# Define the device base names
nm_np_bases = ["{}_D", "{}_G", "{}_S", "{}_B"]
npn_pnp_bases = ["{}_C", "{}_B", "{}_E"]
c_r_l_i_bases = ["{}_P", "{}_N"]
xor_bases = ["{}_A", "{}_B", "{}_VDD", "{}_VSS", "{}_Y"]
pfd_bases = ["{}_A", "{}_B", "{}_QA", "{}_QB", "{}_VDD", "{}_VSS"]
inverter_bases = ["{}_A", "{}_Q", "{}_VDD", "{}_VSS"]
transmission_gate_bases = ["{}_A", "{}_B", "{}_C", "{}_VDD", "{}_VSS"]

# Initialize the list of NM, PM, C, R, L, I, VIN, VB, VOUT devices, and additional entries
devices = []
for prefix in ["NM", "PM"]:
    for i in range(1, 26):
        devices.append(f"{prefix}{i}")
        for base in nm_np_bases:
            devices.append(base.format(f"{prefix}{i}"))

for prefix in ["NPN", "PNP"]:
    for i in range(1, 26):
        devices.append(f"{prefix}{i}")
        for base in npn_pnp_bases:
            devices.append(base.format(f"{prefix}{i}"))

for i in range(1, 26):
    devices.append(f"R{i}")
    for base in c_r_l_i_bases:
        devices.append(base.format(f"R{i}"))

for i in range(1, 26):
    devices.append(f"C{i}")
    for base in c_r_l_i_bases:
        devices.append(base.format(f"C{i}"))

for i in range(1, 26):
    devices.append(f"L{i}")
    for base in c_r_l_i_bases:
        devices.append(base.format(f"L{i}"))

for i in range(1, 26):
    devices.append(f"DIO{i}")
    for base in c_r_l_i_bases:
        devices.append(base.format(f"DIO{i}"))

for i in range(1, 6):
    devices.append(f"XOR{i}")
    for base in xor_bases:
        devices.append(base.format(f"XOR{i}"))

for i in range(1, 6):
    devices.append(f"PFD{i}")
    for base in pfd_bases:
        devices.append(base.format(f"PFD{i}"))

for i in range(1, 11):
    devices.append(f"INVERTER{i}")
    for base in inverter_bases:
        devices.append(base.format(f"INVERTER{i}"))

for i in range(1, 11):
    devices.append(f"TRANSMISSION_GATE{i}")
    for base in transmission_gate_bases:
        devices.append(base.format(f"TRANSMISSION_GATE{i}"))

# for i in range(1, 26):
#     devices.append(f"I{i}")
#     for base in c_r_l_i_bases:
#         devices.append(base.format(f"I{i}"))

for i in range(1, 6):
    devices.append(f"VIN{i}")

for i in range(1, 6):
    devices.append(f"IIN{i}")

for i in range(1, 6):
    devices.append(f"VOUT{i}")

for i in range(1, 6):
    devices.append(f"IOUT{i}")

for i in range(1, 11):
    devices.append(f"VB{i}")

for i in range(1, 11):
    devices.append(f"IB{i}")

for i in range(1, 6):
    devices.append(f"VCONT{i}")

for i in range(1, 6):
    devices.append(f"VCLK{i}")

for i in range(1, 6):
    devices.append(f"VCM{i}")

for i in range(1, 6):
    devices.append(f"VREF{i}")

for i in range(1, 6):
    devices.append(f"IREF{i}")

for i in range(1, 6):
    devices.append(f"VRF{i}")

for i in range(1, 6):
    devices.append(f"VLO{i}")

for i in range(1, 6):
    devices.append(f"VIF{i}")

for i in range(1, 6):
    devices.append(f"VBB{i}")

for i in range(1, 3):
    devices.append(f"LOGICA{i}")

for i in range(1, 3):
    devices.append(f"LOGICB{i}")

for i in range(1, 3):
    devices.append(f"LOGICD{i}")

for i in range(1, 3):
    devices.append(f"LOGICF{i}")

for i in range(1, 3):
    devices.append(f"LOGICG{i}")

for i in range(1, 3):
    devices.append(f"LOGICQ{i}")

for i in range(1, 3):
    devices.append(f"LOGICQA{i}")

for i in range(1, 3):
    devices.append(f"LOGICQB{i}")

# Adding the additional entries
additional_entries = ["VDD", "VSS", "TRUNCATE"]
devices.extend(additional_entries)

# Create a mapping from device names to integers and vice versa
stoi = { device: i for i, device in enumerate(devices) }
itos = { i:device for i,device in enumerate(devices) }
vocab_size = len(devices)

# Print the results
print("Devices in order:", devices)
print("Vocabulary size:", len(devices))
print("Device to index mapping:", stoi)
print("Index to device mapping:", itos)

encode = lambda s: [stoi[c] for c in s] # encoder: take a string, output a list of integers
# decode = lambda l: ''.join([itos[i] for i in l]) # decoder: take a list of integers, output a string
decode = lambda l: '->'.join([itos[i] for i in l]) + '->'

# sequence = ['VDD', 'R2_P', 'R2', 'R2_N', 'VOUT1', 'NM1_D', 'NM1', 'NM1_G', 'VIN1', 'NM1_G', 'NM1', 'NM1_S', 'R1_P', 'R1', 'R1_N', 'VSS', 'NM1_B', 'NM1', 'NM1_B', 'VSS', 'R1_N', 'R1', 'R1_P', 'NM1_S', 'NM1', 'NM1_D', 'VOUT1', 'R2_N', 'R2', 'R2_P', 'VDD']
# print(encode(sequence))
# print(decode(encode(sequence)))
# sequence = np.load('SequenceAll.npy')
# data = torch.tensor(encode(sequence.flatten()), dtype=torch.long).view(sequence.shape)
# print(data.shape)
# n = int(0.9*len(data)) # first 90% will be train, rest val
# train_data = data[:n]
# val_data = data[n:]
# # torch.save(val_data, 'val_data.pt')
# # val_data = torch.load('val_data.pt')

# print(train_data.shape)
# print(val_data.shape)

# Shuffle the data and split it
# n = len(data)
# indices = torch.randperm(n)
# train_size = int(0.9 * n)

# # Split indices for train and validation
# train_indices = indices[:train_size]
# val_indices = indices[train_size:]

# # Select rows based on indices
# train_data = data[train_indices]
# val_data = data[val_indices]
# print(val_indices)
train_data = np.load(Trainingdata)
train_data = torch.tensor(encode(train_data.flatten()), dtype=torch.long).view(train_data.shape)
val_data = np.load(Validationdata)
val_data = torch.tensor(encode(val_data.flatten()), dtype=torch.long).view(val_data.shape)
print(train_data.shape)
print(val_data.shape)

# data loading
# def get_batch(split):
#     # generate a small batch of data of inputs x and targets y
#     data = train_data if split == 'train' else val_data
#     ix = torch.randint(len(data) - block_size, (batch_size,))
#     # print(ix)
#     x = torch.stack([data[i:i+block_size] for i in ix])
#     # print(x.shape)
#     y = torch.stack([data[i+1:i+block_size+1] for i in ix])
#     # print(y.shape)
#     x, y = x.to(device), y.to(device)
#     # print(x)
#     # print(y)
#     return x, y
def get_batch(split):
    # generate a small batch of data of inputs x and targets y
    data = train_data if split == 'train' else val_data
    ix = torch.randint(len(data), (batch_size,))
    # print(ix)
    x = torch.stack([data[i,:-1] for i in ix])
    # print(x.shape)
    y = torch.stack([data[i,1:] for i in ix])
    # print(y.shape)
    x, y = x.to(device), y.to(device)
    # print(x)
    # print(y)
    return x, y

@torch.no_grad()
def estimate_loss():
    out = {}
    model.eval()
    for split in ['train', 'val']:
        losses = torch.zeros(eval_iters)
        for k in range(eval_iters):
            X, Y = get_batch(split)
            logits, loss = model(X, Y)
            losses[k] = loss.item()
        out[split] = losses.mean()
    model.train()
    return out

class Head(nn.Module):
    """ one head of self-attention """

    def __init__(self, head_size):
        super().__init__()
        self.key = nn.Linear(n_embd, head_size, bias=False)
        self.query = nn.Linear(n_embd, head_size, bias=False)
        self.value = nn.Linear(n_embd, head_size, bias=False)
        self.register_buffer('tril', torch.tril(torch.ones(block_size, block_size)))

        self.dropout = nn.Dropout(dropout)

    def forward(self, x):
        # input of size (batch, time-step, channels)
        # output of size (batch, time-step, head size)
        B,T,C = x.shape
        k = self.key(x)   # (B,T,hs)
        q = self.query(x) # (B,T,hs)
        # compute attention scores ("affinities")
        wei = q @ k.transpose(-2,-1) * k.shape[-1]**-0.5 # (B, T, hs) @ (B, hs, T) -> (B, T, T)
        wei = wei.masked_fill(self.tril[:T, :T] == 0, float('-inf')) # (B, T, T)
        wei = F.softmax(wei, dim=-1) # (B, T, T)
        wei = self.dropout(wei)
        # perform the weighted aggregation of the values
        v = self.value(x) # (B,T,hs)
        out = wei @ v # (B, T, T) @ (B, T, hs) -> (B, T, hs)
        return out

class MultiHeadAttention(nn.Module):
    """ multiple heads of self-attention in parallel """

    def __init__(self, num_heads, head_size):
        super().__init__()
        self.heads = nn.ModuleList([Head(head_size) for _ in range(num_heads)])
        self.proj = nn.Linear(head_size * num_heads, n_embd)
        self.dropout = nn.Dropout(dropout)

    def forward(self, x):
        out = torch.cat([h(x) for h in self.heads], dim=-1)
        out = self.dropout(self.proj(out))
        return out

class FeedFoward(nn.Module):
    """ a simple linear layer followed by a non-linearity """

    def __init__(self, n_embd):
        super().__init__()
        self.net = nn.Sequential(
            nn.Linear(n_embd, 4 * n_embd),
            nn.ReLU(),
            nn.Linear(4 * n_embd, n_embd),
            nn.Dropout(dropout),
        )

    def forward(self, x):
        return self.net(x)

class Block(nn.Module):
    """ Transformer block: communication followed by computation """

    def __init__(self, n_embd, n_head):
        # n_embd: embedding dimension, n_head: the number of heads we'd like
        super().__init__()
        head_size = n_embd // n_head
        self.sa = MultiHeadAttention(n_head, head_size)
        self.ffwd = FeedFoward(n_embd)
        self.ln1 = nn.LayerNorm(n_embd)
        self.ln2 = nn.LayerNorm(n_embd)

    def forward(self, x):
        x = x + self.sa(self.ln1(x))
        x = x + self.ffwd(self.ln2(x))
        return x

class GPTLanguageModel(nn.Module):

    def __init__(self):
        super().__init__()
        # each token directly reads off the logits for the next token from a lookup table
        self.token_embedding_table = nn.Embedding(vocab_size, n_embd)
        self.position_embedding_table = nn.Embedding(block_size, n_embd)
        self.blocks = nn.Sequential(*[Block(n_embd, n_head=n_head) for _ in range(n_layer)])
        self.ln_f = nn.LayerNorm(n_embd) # final layer norm
        self.lm_head = nn.Linear(n_embd, vocab_size)

        # better init, not covered in the original GPT video, but important, will cover in followup video
        self.apply(self._init_weights)

    def _init_weights(self, module):
        if isinstance(module, nn.Linear):
            torch.nn.init.normal_(module.weight, mean=0.0, std=0.02)
            if module.bias is not None:
                torch.nn.init.zeros_(module.bias)
        elif isinstance(module, nn.Embedding):
            torch.nn.init.normal_(module.weight, mean=0.0, std=0.02)

    def forward(self, idx, targets=None):
        B, T = idx.shape

        # idx and targets are both (B,T) tensor of integers
        tok_emb = self.token_embedding_table(idx) # (B,T,C)
        pos_emb = self.position_embedding_table(torch.arange(T, device=device)) # (T,C)
        x = tok_emb + pos_emb # (B,T,C)
        x = self.blocks(x) # (B,T,C)
        x = self.ln_f(x) # (B,T,C)
        logits = self.lm_head(x) # (B,T,vocab_size)
        # print(logits.shape)

        if targets is None:
            loss = None
        else:
            B, T, C = logits.shape
            logits = logits.view(B*T, C)
            targets = targets.view(B*T)
            loss = F.cross_entropy(logits, targets)

        return logits, loss

    def generate(self, idx, max_new_tokens):
        # idx is (B, T) array of indices in the current context
        for _ in range(max_new_tokens):
            # crop idx to the last block_size tokens
            idx_cond = idx[:, -block_size:]
            # get the predictions
            logits, loss = self(idx_cond)
            # focus only on the last time step
            logits = logits[:, -1, :] # becomes (B, C)
            # apply softmax to get probabilities
            probs = F.softmax(logits, dim=-1) # (B, C)
            # sample from the distribution
            idx_next = torch.multinomial(probs, num_samples=1) # (B, 1)
            # append sampled index to the running sequence
            idx = torch.cat((idx, idx_next), dim=1) # (B, T+1)
        return idx

model = GPTLanguageModel()
m = model.to(device)
# print the number of parameters in the model
print(sum(p.numel() for p in m.parameters())/1e6, 'M parameters')

# create a PyTorch optimizer
optimizer = torch.optim.AdamW(model.parameters(), lr=learning_rate)
val_loss = float('inf')

# Open a CSV file to write the losses
csv_file = open(filename+'.csv', mode='w', newline='')
csv_writer = csv.writer(csv_file)
csv_writer.writerow(['Step', 'Train Loss', 'Validation Loss'])

for iter in range(max_iters):
    torch.cuda.empty_cache()

    # every once in a while evaluate the loss on train and val sets
    if iter % eval_interval == 0 or iter == max_iters - 1:
        losses = estimate_loss()
        print(f"step {iter}: train loss {losses['train']:.4f}, val loss {losses['val']:.4f}")
        csv_writer.writerow([iter, losses['train'].item(), losses['val'].item()]) # Write losses to CSV
        if losses['val'] < val_loss:
            savemodel_name = filename + '.pth'
            torch.save(model.state_dict(), savemodel_name)
            val_loss = losses['val']

    # sample a batch of data
    xb, yb = get_batch('train')
    # print(xb.shape)
    # print(yb.shape)

    # evaluate the loss
    logits, loss = model(xb, yb)
    optimizer.zero_grad(set_to_none=True)
    loss.backward()
    optimizer.step()

# generate from the model
# context = torch.zeros((1, 1), dtype=torch.long, device=device)
# value = decode('VDD')

# model.load_state_dict(torch.load(savemodel_name))
# run = 10
# for i in range(run):
#     context = torch.full((1, 1), 896, dtype=torch.long, device=device)
#     save_dir = 'generation_result/run'+str(i)+'.txt'
#     # save_dir = str(i) + 'run.txt'
#     open(save_dir, 'w').write(decode(m.generate(context, max_new_tokens=512)[0].tolist()))
#     # print(decode(m.generate(context, max_new_tokens=256)[0].tolist()))
# #open('more.txt', 'w').write(decode(m.generate(context, max_new_tokens=10000)[0].tolist()))
