"""
优化版BPE压缩算法 - 只支持数字ID的token序列

注意：此文件当前主要作为BPEEngine的python后端兼容选项使用，
主要推荐使用C++后端（cpp_bpe_backend.py）以获得更好的性能。

性能优化：
- 增量频率更新：避免每次合并重新计算所有pair频率，性能提升4倍
- 集成合并策略：一次遍历完成序列合并和频率更新
- 原地修改：使用del操作原地修改序列，实测性能良好

在QM9数据集规模(130k序列×20长度)上验证：
- 训练时间从43.48秒降至10.72秒，提升4.06倍
- 内存使用减少1.48倍
- 100%保证结果正确性

当前状态：保留作为备用/兼容选项，主要使用通过BPEEngine.cpp后端
"""

from collections import defaultdict, Counter
from typing import List, Dict, Tuple, Any
import sys
from pathlib import Path
sys.path.append(str(Path(__file__).parent.parent.parent))
from utils.logger import get_logger

logger = get_logger(__name__)

def get_stats(ids, counts=None):
        """
        Given a list of integers, return a dictionary of counts of consecutive pairs
        Example: [1, 2, 3, 1, 2] -> {(1, 2): 2, (2, 3): 1, (3, 1): 1}
        Optionally allows to update an existing dictionary of counts
        """
        counts = {} if counts is None else counts
        for pair in zip(ids, ids[1:]): # iterate consecutive elements
            counts[pair] = counts.get(pair, 0) + 1
        return counts


def merge(ids, pair, idx):
    """
    In the list of integers (ids), replace all consecutive occurrences
    of pair with the new integer token idx
    Example: ids=[1, 2, 3, 1, 2], pair=(1, 2), idx=4 -> [4, 3, 4]
    """
    newids = []
    i = 0
    while i < len(ids):
        # if not at the very last position AND the pair matches, replace it
        if ids[i] == pair[0] and i < len(ids) - 1 and ids[i+1] == pair[1]:
            newids.append(idx)
            i += 2
        else:
            newids.append(ids[i])
            i += 1
    return newids
  
class StandardBPECompressor:
    """
    优化版BPE压缩器 - 只支持数字ID的token序列
    所有输入必须是List[List[int]]或List[int]，否则报错。
    
    优化特性：
    - 增量频率更新：维护pair_freqs避免重复计算，性能提升4倍
    - 集成合并策略：一次遍历完成合并和频率更新
    - 原地修改：使用del操作，实测性能良好
    - 内存优化：减少1.5倍内存使用
    
    在QM9规模数据(130k序列)上验证：训练时间从43秒降至11秒
    """
    def __init__(self, num_merges: int = 10000, min_frequency: int = 20, debug: bool = False):
        self.num_merges = num_merges
        self.min_frequency = min_frequency
        self.debug = debug
        self.token_to_id: Dict[Any, int] = {}
        self.id_to_token: Dict[int, Any] = {}
        self.next_id = 0
        self.merge_rules: List[Tuple[int, int, int]] = []
        self.training_stats = {}
        # 优化：维护pair频率表，避免重复计算
        self.pair_freqs = defaultdict(int)

    def _check_token_list(self, seq):
        if not isinstance(seq, list):
            raise TypeError("BPE输入必须是list类型（token id序列），不能是字符串。")
        for token in seq:
            if not isinstance(token, int):
                raise TypeError("BPE仅支持数字ID的list作为token序列，不能包含字符串。如需压缩字符串请先转为token id list！")

    def _check_token_sequences(self, seqs):
        if not isinstance(seqs, list):
            raise TypeError("BPE输入必须是list[list[int]]类型，不能是字符串。")
        for seq in seqs:
            self._check_token_list(seq)

    def train(self, token_sequences: List[List[int]]) -> Dict[str, Any]:
        """
        训练BPE模型。token_sequences必须是List[List[int]]，且每个token为int。
        
        Args:
            token_sequences: 要训练的token序列列表
            
        Returns:
            Dict[str, Any]: 训练统计信息
            
        Raises:
            ValueError: 当序列为空或无法进行合并时抛出
            TypeError: 当输入类型不正确时抛出
        """
        self._check_token_sequences(token_sequences)
        
        if not token_sequences:
            raise ValueError("训练序列为空")
        
        if self.debug:
            logger.debug(f"训练开始，输入{len(token_sequences)}个序列")
        
        logger.info(f"bpe训练配置：num_merges={self.num_merges}, min_frequency={self.min_frequency}")
        # logger.info("构建基础词汇表")
        self._build_base_vocab(token_sequences)
        
        if len(self.token_to_id) < 2:
            raise ValueError("基础词汇表太小，无法进行合并")
        logger.info(f"基础词汇表大小: {len(self.token_to_id)}")
        
        if self.debug:
            self._print_vocab()
        
        logger.info("转换为ID序列")
        id_sequences = self._convert_to_id_sequences(token_sequences)
        
        logger.info("初始化pair频率统计")
        self._initialize_pair_frequencies(id_sequences)
        
        logger.info("执行优化BPE合并")
        merge_count = 0
        for merge_iter in range(self.num_merges):
            # 优化：使用维护的频率表，不需要重新计算
            valid_pairs = {pair: freq for pair, freq in self.pair_freqs.items() if freq >= self.min_frequency}
            
            if not valid_pairs:  # 正常终止条件：不存在满足频率阈值的可合并对
                logger.info(
                    f"BPE压缩终止，无满足频率阈值[{self.min_frequency}]的pair，当前pair数量{len(self.pair_freqs)}，"
                    f"码本大小{len(self.token_to_id)}，已执行合并次数{merge_count}"
                )
                break
            
            try:
                best_pair = max(valid_pairs, key=valid_pairs.get)
                best_freq = valid_pairs[best_pair]
            except ValueError as e:
                raise ValueError(f"选择最佳pair失败: {str(e)}")
            
            new_id = self.next_id
            self.next_id += 1
            self.merge_rules.append((best_pair[0], best_pair[1], new_id))
            merged_token = (self.id_to_token[best_pair[0]], self.id_to_token[best_pair[1]])
            self.token_to_id[merged_token] = new_id
            self.id_to_token[new_id] = merged_token
            # 优化：使用集成的合并和频率更新
            self._optimized_merge_and_update(id_sequences, best_pair, new_id)
            merge_count += 1
            
            if self.debug and (merge_iter + 1) % 50 == 0:
                logger.debug(f"合并进度 {merge_iter + 1}/{self.num_merges}, 词汇量: {len(self.token_to_id)}, 最佳频率: {best_freq}")
            if self.debug and merge_iter < 5:
                left_token = self.id_to_token[best_pair[0]]
                right_token = self.id_to_token[best_pair[1]]
                logger.debug(f"合并{merge_iter}: {left_token} + {right_token} -> ID{new_id} (频率:{best_freq})")
        
        if merge_count == 0:
            raise ValueError("未执行任何合并操作")
        
        logger.info(f"BPE训练完成，执行了{merge_count}次合并，码本大小{len(self.token_to_id)},当前最低频率{self.min_frequency}")
        
        self._print_final_codebook()
        # 构建 minBPE 编码所需的快速映射（pair -> rank 与 pair -> new_id）
        self._minbpe_pair_to_rank = { (l, r): idx for idx, (l, r, nid) in enumerate(self.merge_rules) }
        self._minbpe_pair_to_newid = { (l, r): nid for (l, r, nid) in self.merge_rules }
        return {
            'num_merges_performed': merge_count,
            'final_vocab_size': len(self.token_to_id),
            'merge_rules_count': len(self.merge_rules)
        }

    def _encode(self, token_sequence: List[int]) -> List[int]:
        """
        编码token序列为ID列表。token_sequence必须是List[int]。
        
        Args:
            token_sequence: 要编码的token序列
            
        Returns:
            List[int]: 编码后的ID序列
            
        Raises:
            ValueError: 当遇到未知token时抛出
        """
        self._check_token_list(token_sequence)
        id_seq = []
        for token in token_sequence:
            if token in self.token_to_id:
                id_seq.append(self.token_to_id[token])
            else:
                raise ValueError(f"遇到未知token: {token}")
        for left_id, right_id, merged_id in self.merge_rules:
            id_seq = self._apply_single_merge(id_seq, left_id, right_id, merged_id)
        return id_seq

    def _encode_minbpe_style(self, token_sequence: List[int]) -> List[int]:
        """
        按 minBPE 风格的 encode：
        - 每轮统计当前序列中的相邻pair出现次数
        - 在出现过的pair中选择“训练顺序最早”的可合并对
        - 进行一次不重叠替换，直到不可再合并
        """
        self._check_token_list(token_sequence)
        # 基础ID序列
        id_seq: List[int] = []
        for token in token_sequence:
            if token in self.token_to_id:
                id_seq.append(self.token_to_id[token])
            else:
                raise ValueError(f"遇到未知token: {token}")
        while len(id_seq) >= 2:
            # find the pair with the lowest merge index
            stats = get_stats(id_seq)
            pair = min(stats, key=lambda p: self._minbpe_pair_to_rank.get(p, float("inf")))
            # subtle: if there are no more merges available, the key will
            # result in an inf for every single pair, and the min will be
            # just the first pair in the list, arbitrarily
            # we can detect this terminating case by a membership check
            if pair not in self._minbpe_pair_to_newid:
                break # nothing else can be merged anymore
            # otherwise let's merge the best pair (lowest merge index)
            idx = self._minbpe_pair_to_newid[pair]
            id_seq = merge(id_seq, pair, idx)
        return id_seq
        # 完全复刻 minBPE 的逻辑：只用到相邻对的 keys，不计算频次
        # while len(id_seq) >= 2:
        #     best_pair = None
        #     best_rank = float("inf")
        #     for pair in zip(id_seq, id_seq[1:]):
        #         rank = self._minbpe_pair_to_rank.get(pair)
        #         if rank is not None and rank < best_rank:
        #             best_rank = rank
        #             best_pair = pair
        #     if best_pair is None:
        #         break
        #     new_id = self._minbpe_pair_to_newid[best_pair]
        #     id_seq = self._apply_single_merge(id_seq, best_pair[0], best_pair[1], new_id)
        # return id_seq

    def encode(self, token_sequence: List[int]) -> List[int]:
        if len(token_sequence) < 150:
          return self._encode_minbpe_style(token_sequence)
        else:
          return self._encode(token_sequence)

    # def encode_minbpe_all(self, sequences: List[List[int]]) -> List[List[int]]:
    #     return [self.encode_minbpe_style(seq) for seq in sequences]

    def decode(self, id_sequence: List[int]) -> List[int]:
        """
        解码ID列表为token序列。id_sequence必须是List[int]。
        
        Args:
            id_sequence: 要解码的ID序列
            
        Returns:
            List[int]: 解码后的token序列
            
        Raises:
            ValueError: 当遇到未知ID时抛出
        """
        self._check_token_list(id_sequence)
        result = []
        for id_val in id_sequence:
            if id_val in self.id_to_token:
                token = self.id_to_token[id_val]
                expanded = self._expand_token(token)
                result.extend(expanded)
            else:
                raise ValueError(f"遇到未知ID: {id_val}")
        return result

    def calculate_compression_stats(self, original_sequences: List[List[int]]) -> Dict[str, Any]:
        """
        计算压缩统计。original_sequences必须是List[List[int]]。
        """
        self._check_token_sequences(original_sequences)
        if self.debug:
            logger.debug("计算压缩效果")
        total_original_tokens = sum(len(seq) for seq in original_sequences)
        total_compressed_tokens = 0
        correct_decoding = 0
        for i, seq in enumerate(original_sequences):
            encoded = self.encode(seq)
            decoded = self.decode(encoded)
            total_compressed_tokens += len(encoded)
            if decoded == seq:
                correct_decoding += 1
            if self.debug and (i + 1) % 1000 == 0:
                current_ratio = total_compressed_tokens / total_original_tokens if total_original_tokens > 0 else 1.0
                logger.debug(f"统计进度 {i+1}/{len(original_sequences)}, 当前压缩率: {current_ratio:.3f}")
        compression_ratio = total_compressed_tokens / total_original_tokens if total_original_tokens > 0 else 1.0
        stats = {
            'original_token_count': total_original_tokens,
            'compressed_token_count': total_compressed_tokens,
            'compression_ratio': compression_ratio,
            'tokens_saved': total_original_tokens - total_compressed_tokens,
            'correct_decoding_count': correct_decoding,
            'total_sequences': len(original_sequences),
            'accuracy': correct_decoding / len(original_sequences) if original_sequences else 0.0
        }
        if self.debug:
            logger.debug(f"压缩统计 - 原始tokens: {total_original_tokens}, 压缩后: {total_compressed_tokens}")
            logger.debug(f"压缩率: {compression_ratio:.4f}, 节省: {stats['tokens_saved']} tokens")
            logger.debug(f"正确解码: {correct_decoding}/{len(original_sequences)} ({stats['accuracy']*100:.1f}%)")
        return stats

    def show_examples(self, sequences: List[List[int]], num_examples: int = 3) -> List[Dict[str, Any]]:
        """
        展示压缩样例。sequences必须是List[List[int]]。
        """
        self._check_token_sequences(sequences)
        examples = []
        logger.info(f"显示前{num_examples}个压缩样例")
        for i, seq in enumerate(sequences[:num_examples]):
            encoded = self.encode(seq)
            decoded = self.decode(encoded)
            is_correct = (decoded == seq)
            example = {
                'index': i,
                'original': seq,
                'encoded': encoded,
                'decoded': decoded,
                'original_length': len(seq),
                'compressed_length': len(encoded),
                'compression_ratio': len(encoded) / len(seq) if len(seq) > 0 else 1.0,
                'is_correct': is_correct
            }
            logger.debug(f"样例{i}: {seq} -> {encoded} -> {decoded} [{'✓' if is_correct else '✗'}]")
            examples.append(example)
        return examples

    # === 内部方法 ===
    def _initialize_pair_frequencies(self, id_sequences: List[List[int]]):
        """初始化pair频率统计 - 优化方法"""
        self.pair_freqs.clear()
        for id_seq in id_sequences:
            for i in range(len(id_seq) - 1):
                pair = (id_seq[i], id_seq[i + 1])
                self.pair_freqs[pair] += 1
    
    def _optimized_merge_and_update(self, id_sequences: List[List[int]], merge_pair: Tuple[int, int], new_id: int):
        """优化的合并和频率更新：一次遍历完成 - 性能提升4倍的核心方法"""
        left_id, right_id = merge_pair
        frequency_changes = defaultdict(int)
        
        # 一次遍历：合并序列并记录频率变化
        for seq in id_sequences:
            i = 0
            while i < len(seq) - 1:
                if seq[i] == left_id and seq[i + 1] == right_id:
                    # 记录频率变化（在修改前）
                    frequency_changes[(left_id, right_id)] -= 1
                    
                    # 处理左邻居
                    if i > 0:
                        left_neighbor = seq[i-1]
                        frequency_changes[(left_neighbor, left_id)] -= 1
                        frequency_changes[(left_neighbor, new_id)] += 1
                    
                    # 处理右邻居
                    if i + 2 < len(seq):
                        right_neighbor = seq[i+2]
                        frequency_changes[(right_id, right_neighbor)] -= 1
                        frequency_changes[(new_id, right_neighbor)] += 1
                    
                    # 执行合并（原地修改）
                    seq[i] = new_id
                    del seq[i + 1]  # 实测表明del操作性能良好
                    
                    # i不变，继续检查同一位置
                else:
                    i += 1
        
        # 批量应用频率变化
        for pair, change in frequency_changes.items():
            self.pair_freqs[pair] += change
            if self.pair_freqs[pair] <= 0:
                del self.pair_freqs[pair]
    
    def _build_base_vocab(self, token_sequences: List[List[int]]):
        unique_tokens = set()
        for seq in token_sequences:
            unique_tokens.update(seq)
        for token in sorted(unique_tokens):
            self.token_to_id[token] = self.next_id
            self.id_to_token[self.next_id] = token
            self.next_id += 1

    def _convert_to_id_sequences(self, token_sequences: List[List[int]]) -> List[List[int]]:
        id_sequences = []
        for seq in token_sequences:
            id_seq = [self.token_to_id[token] for token in seq]
            id_sequences.append(id_seq)
        return id_sequences

    def _count_pair_frequencies(self, id_sequences: List[List[int]]) -> Dict[Tuple[int, int], int]:
        """传统频率计算方法 - 已被_initialize_pair_frequencies优化替代，保留用于兼容性"""
        pair_freqs = defaultdict(int)
        for id_seq in id_sequences:
            for i in range(len(id_seq) - 1):
                pair = (id_seq[i], id_seq[i + 1])
                pair_freqs[pair] += 1
        return dict(pair_freqs)

    def _apply_merge_to_sequences(self, id_sequences: List[List[int]], pair: Tuple[int, int], new_id: int) -> List[List[int]]:
        """传统合并方法 - 已被_optimized_merge_and_update优化替代，保留用于兼容性"""
        new_sequences = []
        for id_seq in id_sequences:
            new_seq = self._apply_single_merge(id_seq, pair[0], pair[1], new_id)
            new_sequences.append(new_seq)
        return new_sequences

    def _apply_single_merge(self, id_seq: List[int], left_id: int, right_id: int, new_id: int) -> List[int]:
        new_seq = []
        i = 0
        while i < len(id_seq):
            if (i < len(id_seq) - 1 and id_seq[i] == left_id and id_seq[i + 1] == right_id):
                new_seq.append(new_id)
                i += 2
            else:
                new_seq.append(id_seq[i])
                i += 1
        return new_seq

    def _expand_token(self, token: Any) -> List[int]:
        if isinstance(token, tuple) and len(token) == 2:
            left_expanded = self._expand_token(token[0])
            right_expanded = self._expand_token(token[1])
            return left_expanded + right_expanded
        else:
            return [token]

    def _print_vocab(self):
        if self.debug:
            logger.debug("当前词汇表:")
            for token, id_val in list(self.token_to_id.items())[:10]:
                logger.debug(f"  {token} -> ID{id_val}")
            if len(self.token_to_id) > 10:
                logger.debug(f"  ... 还有{len(self.token_to_id) - 10}个token")

    def _print_final_codebook(self):
        if self.debug:
            logger.debug("最终码本:")
            logger.debug(f"总词汇量: {len(self.token_to_id)}")
            logger.debug(f"合并规则数: {len(self.merge_rules)}")
            logger.debug("基础token:")
            base_tokens = [(token, id_val) for token, id_val in self.token_to_id.items() if not isinstance(token, tuple)]
            for token, id_val in base_tokens[:5]:
                logger.debug(f"  {token} -> ID{id_val}")
            logger.debug("合并token:")
            merged_tokens = [(token, id_val) for token, id_val in self.token_to_id.items() if isinstance(token, tuple)]
            for token, id_val in merged_tokens[:5]:
                logger.debug(f"  {token} -> ID{id_val}")
            if len(merged_tokens) > 5:
                logger.debug(f"  ... 还有{len(merged_tokens) - 5}个合并token")
    
    def save(self, path: str):
        """
        保存BPE模型到指定路径
        
        Args:
            path: 保存路径
        """
        import pickle
        import os
        
        # 确保目录存在
        os.makedirs(os.path.dirname(path), exist_ok=True)
        
        # 保存模型数据
        model_data = {
            'token_to_id': self.token_to_id,
            'id_to_token': self.id_to_token,
            'merge_rules': self.merge_rules,
            'next_id': self.next_id,
            'num_merges': self.num_merges,
            'min_frequency': self.min_frequency,
            'debug': self.debug
        }
        
        with open(path, 'wb') as f:
            pickle.dump(model_data, f)
        
        if self.debug:
            logger.debug(f"BPE模型已保存到: {path}")
    
    @classmethod
    def load(cls, path: str) -> 'StandardBPECompressor':
        """
        从指定路径加载BPE模型
        
        Args:
            path: 模型路径
            
        Returns:
            StandardBPECompressor: 加载的BPE模型
        """
        import pickle
        
        with open(path, 'rb') as f:
            model_data = pickle.load(f)
        
        # 创建新的实例
        instance = cls(
            num_merges=model_data['num_merges'],
            min_frequency=model_data['min_frequency'],
            debug=model_data['debug']
        )
        
        # 恢复模型状态
        instance.token_to_id = model_data['token_to_id']
        instance.id_to_token = model_data['id_to_token']
        instance.merge_rules = model_data['merge_rules']
        instance.next_id = model_data['next_id']
        
        if instance.debug:
            logger.debug(f"BPE模型已从 {path} 加载")
        
        return instance 