"""
Example of running PyTorch implementation of DDPG on Swimmer.
"""
import copy

import sys
import torch

from gym.envs.mujoco import ReacherEnv

from torch.nn import functional as F

from rlkit.data_management.env_replay_buffer import EnvReplayBuffer
from rlkit.envs.wrappers import NormalizedBoxEnv
from rlkit.exploration_strategies.base import (
    PolicyWrappedWithExplorationStrategy
)
from rlkit.exploration_strategies.ou_strategy import OUStrategy
from rlkit.launchers.launcher_util import setup_logger
from rlkit.samplers.data_collector import MdpPathCollector
from rlkit.torch.networks import ConcatMlp, TanhMlpPolicy
from rlkit.torch.manifold.mrl_ddpg import MRLDDPGTrainer
import rlkit.torch.pytorch_util as ptu
from rlkit.torch.torch_rl_algorithm import TorchBatchRLAlgorithm


def experiment(variant):
    eval_env = NormalizedBoxEnv(ReacherEnv())
    expl_env = NormalizedBoxEnv(ReacherEnv())
    if variant["activation_type"].lower() == "gelu":
        hidden_activation = F.gelu
    else:
        hidden_activation = F.relu
    # Or for a specific version:
    # import gym
    # env = NormalizedBoxEnv(gym.make('HalfCheetah-v1'))
    obs_dim = eval_env.observation_space.low.size
    action_dim = eval_env.action_space.low.size
    qf = ConcatMlp(
        input_size=obs_dim + action_dim,
        output_size=1,
        hidden_activation=hidden_activation,
        **variant['qf_kwargs']
    )
    policy = TanhMlpPolicy(
        input_size=obs_dim,
        output_size=action_dim,
        hidden_activation=hidden_activation,
        manifold_rep_layer=variant["manifold_rep_layer"],
        **variant['policy_kwargs']
    )
    target_qf = copy.deepcopy(qf)
    target_policy = copy.deepcopy(policy)
    eval_path_collector = MdpPathCollector(eval_env, policy)
    exploration_policy = PolicyWrappedWithExplorationStrategy(
        exploration_strategy=OUStrategy(action_space=expl_env.action_space),
        policy=policy,
    )
    expl_path_collector = MdpPathCollector(expl_env, exploration_policy)
    replay_buffer = EnvReplayBuffer(variant['replay_buffer_size'], expl_env)
    trainer = MRLDDPGTrainer(
        qf=qf,
        target_qf=target_qf,
        policy=policy,
        target_policy=target_policy,
        **variant['trainer_kwargs']
    )
    algorithm = TorchBatchRLAlgorithm(
        trainer=trainer,
        exploration_env=expl_env,
        evaluation_env=eval_env,
        exploration_data_collector=expl_path_collector,
        evaluation_data_collector=eval_path_collector,
        replay_buffer=replay_buffer,
        **variant['algorithm_kwargs']
    )
    algorithm.to(ptu.device)
    algorithm.train()


if __name__ == "__main__":
    # noinspection PyTypeChecker
    random_seed = int(sys.argv[1])
    arch_file_path = sys.argv[2]
    manifold_rep_layer = int(sys.argv[3])
    manifold_rep_weight = float(sys.argv[4])
    node_batch_size = int(sys.argv[5])
    activation_type = sys.argv[6]
    num_graph_hops = int(sys.argv[7])
    num_nodes_rep = int(sys.argv[8])
    max_graph_points = int(sys.argv[9])

    width_arch = []
    eval_env = NormalizedBoxEnv(ReacherEnv())
    obs_dim = eval_env.observation_space.low.size
    action_dim = eval_env.action_space.low.size
    print("Env dimensions:")
    print(obs_dim, action_dim)
    action_dim = eval_env.action_space.low.size


    with open(arch_file_path, "r") as arch_file:
        for l in arch_file.readlines():
            l_arr = l.split(",")
            num_hidden = len(l_arr)
            rep_layer_size = int(num_hidden/2)
            for i, hidden_width in enumerate(l_arr):
                width_arch.append(int(hidden_width))


    variant = dict(
        algorithm_kwargs=dict(
            num_epochs=1000,
            num_eval_steps_per_epoch=1000,
            num_trains_per_train_loop=1000,
            num_expl_steps_per_train_loop=1000,
            min_num_steps_before_training=10000,
            max_path_length=1000,
            num_train_loops_per_epoch=1,
            batch_size=128,
            is_manifold_rep_learning=True,
            max_graph_points=max_graph_points
        ),
        trainer_kwargs=dict(
            use_soft_update=True,
            tau=1e-2,
            discount=0.99,
            qf_learning_rate=1e-3,
            policy_learning_rate=1e-4,
            manifold_rep_weight=manifold_rep_weight,
            batch_size_rep=node_batch_size,
            num_graph_hops=num_graph_hops,
            num_nodes_rep=num_nodes_rep
        ),
        qf_kwargs=dict(
            hidden_sizes=[400, 300],
        ),
        policy_kwargs=dict(
            hidden_sizes=width_arch,
        ),
        replay_buffer_size=int(1E6),
        activation_type=activation_type,
        manifold_rep_layer=manifold_rep_layer
    )
    ptu.set_gpu_mode(True)  # optionally set the GPU (default=False)
    print("Seed: " + str(random_seed))
    print("Arch file: " + arch_file_path)
    setup_logger('name-of-experiment', variant=variant)
    torch.manual_seed(random_seed)
    experiment(variant)
