import argparse
import math
import os
import shutil

import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
from scipy.sparse import csr_matrix
from tensorboardX import SummaryWriter
from torch.optim.lr_scheduler import LambdaLR
from torch.utils.data import DataLoader
from tqdm import tqdm

import dgl
from gnn import GNNModel
import torch_geometric.transforms as T
import wandb
from ogb.linkproppred import Evaluator, PygLinkPropPredDataset, DglLinkPropPredDataset
from optim_schedule import ScheduledOptim
from torch_geometric.utils import add_remaining_self_loops, to_undirected
from transformer import TransformerModel

class LinkPredictor(torch.nn.Module):
    def __init__(self, in_channels, hidden_channels, out_channels, num_layers,
                 dropout):
        super(LinkPredictor, self).__init__()

        self.lins = torch.nn.ModuleList()
        self.lins.append(torch.nn.Linear(in_channels, hidden_channels))
        for _ in range(num_layers - 2):
            self.lins.append(torch.nn.Linear(hidden_channels, hidden_channels))
        self.lins.append(torch.nn.Linear(hidden_channels, out_channels))

        self.dropout = dropout

    def reset_parameters(self):
        for lin in self.lins:
            lin.reset_parameters()

    def forward(self, x_i, x_j):
        x = x_i * x_j
        for lin in self.lins[:-1]:
            x = lin(x)
            x = F.relu(x)
            x = F.dropout(x, p=self.dropout, training=self.training)
        x = self.lins[-1](x)
        return torch.sigmoid(x)

class LinkPredictionDataset(torch.utils.data.Dataset):
    def __init__(self, dgl_data, ego_graphs, edge_idx):
        super(LinkPredictionDataset).__init__()
        self.graph = dgl_data
        self.ego_graphs = ego_graphs
        self.edge_idx = edge_idx['edge']

    def __len__(self):
        return len(self.edge_idx)

    def __getitem__(self, idx):
        ret = []
        for ego in torch.cat([self.edge_idx[idx], torch.randint(0, len(self.ego_graphs), (1,), dtype=torch.long)]):
            nids = self.ego_graphs[ego]
            subg = self.graph.subgraph(nids)
            ret.append(ego)
            ret.append(subg)
        return ret
    
class LinkPredictionTestDataset(torch.utils.data.Dataset):
    def __init__(self, dgl_data, ego_graphs, edge_idx):
        super(LinkPredictionTestDataset).__init__()
        self.graph = dgl_data
        self.ego_graphs = ego_graphs
        self.edge_idx = edge_idx['edge']
        self.edge_idx_neg = edge_idx['edge_neg']

    def __len__(self):
        return len(self.edge_idx)

    def __getitem__(self, idx):
        ret = []
        for ego in torch.cat([self.edge_idx[idx], self.edge_idx_neg[idx]]):
            nids = self.ego_graphs[ego]
            subg = self.graph.subgraph(nids)
            ret.append(ego)
            ret.append(subg)
        return ret

def make_batcher():
    def batcher(batch):
        ret = list(zip(*batch))
        for i in range(len(ret)):
            if i % 2 == 0:
                ret[i] = torch.stack(ret[i]).long()
            else:
                ret[i] = dgl.batch(ret[i])
        return ret

    return batcher

def train(model, predictor, train_loader, args, optimizer, device):
    model.train()
    predictor.train()
    tqdm_loader = tqdm(train_loader)

    total_loss: float = 0
    total_examples: int = 0
    for t, batch in enumerate(tqdm_loader):
        optimizer.zero_grad()
        batch = [x.to(device) for x in batch]
        i, g, i_pos, g_pos, i_neg, g_neg = batch
        h = model(g)
        h_pos = model(g_pos)
        h_neg = model(g_neg)

        pos_out = predictor(h, h_pos)
        pos_loss = -torch.log(pos_out + 1e-15).mean()

        neg_out = predictor(h, h_neg)
        neg_loss = -torch.log(1 - neg_out + 1e-15).mean()

        loss = pos_loss + neg_loss
        tqdm_loader.set_description(f"loss={loss.item():.2f}")
        loss.backward()
        optimizer.step()

        num_examples = pos_out.size(0)
        total_loss += loss.item() * num_examples
        total_examples += num_examples
        if (t + 1) % 400 == 0:
            yield total_loss / total_examples
    
    yield total_loss / total_examples

@torch.no_grad()
def test(model, predictor, dataloader, evaluator, device):
    model.eval()
    predictor.eval()

    preds = []
    preds_neg = []
    for batch in tqdm(dataloader):
        # pos_i1, pos_g1, pos_i2, pos_g2, neg_i1, neg_g1, neg_i2, neg_g2
        batch = [x.to(device) for x in batch]
        hs = [model(batch[2 * i + 1]) for i in range(4)]
        preds += [predictor(hs[0], hs[1]).squeeze().cpu()]
        preds_neg += [predictor(hs[2], hs[3]).squeeze().cpu()]
    pred = torch.cat(preds, dim=0)
    pred_neg = torch.cat(preds_neg, dim=0)

    results = {}
    for K in [10, 20, 50, 100]:
        evaluator.K = K
        hits = evaluator.eval({
            'y_pred_pos': pred,
            'y_pred_neg': pred_neg,
        })[f'hits@{K}']

        results[f'Hits@{K}'] = hits

    return results

def get_exp_name(dataset, para_dic, input_exp_name):
    para_name = '_'.join([dataset] + [key + str(value) for key, value in para_dic.items()])
    exp_name = para_name + '_' + input_exp_name

    if os.path.exists('runs/' + exp_name):
        shutil.rmtree('runs/' + exp_name)

    return exp_name

def get_lr(optimizer):
    for param_group in optimizer.param_groups:
        return param_group['lr']

def get_positional_embedding(max_len, d_model):
    pe = torch.zeros(max_len, d_model)
    position = torch.arange(0, max_len, dtype=torch.float).unsqueeze(1)
    div_term = torch.exp(torch.arange(0, d_model, 2).float() * (-math.log(10000.0) / d_model))
    pe[:, 0::2] = torch.sin(position * div_term)
    pe[:, 1::2] = torch.cos(position * div_term)
    pe = pe.unsqueeze(0).transpose(0, 1)
    return pe

def main():
    parser = argparse.ArgumentParser(description='OGBN (GNN)')
    parser.add_argument('--device', type=int, default=0)
    parser.add_argument('--dataset', type=str, default='collab')
    parser.add_argument('--model', type=str, default='gcn')
    parser.add_argument('--log_steps', type=int, default=1)
    parser.add_argument('--num_layers', type=int, default=3)
    parser.add_argument('--ego_size', type=int, default=64)
    parser.add_argument('--hidden_size', type=int, default=64)
    parser.add_argument('--hidden_dropout', type=float, default=0.0)
    parser.add_argument('--weight_decay', type=float, default=0.0)
    parser.add_argument('--lr', type=float, default=0.001)
    parser.add_argument('--lr_scale', type=float, default=1.0)
    parser.add_argument('--epochs', type=int, default=500)
    parser.add_argument('--early_stopping', type=int, default=10)
    parser.add_argument('--batch_size', type=int, default=512)
    parser.add_argument('--eval_batch_size', type=int, default=2048)
    parser.add_argument('--num_workers', type=int, default=16, help='number of workers')
    parser.add_argument('--batch_norm', type=int, default=1)
    parser.add_argument('--residual', type=int, default=0)
    parser.add_argument('--norm', type=str, default='both')
    parser.add_argument('--runs', type=int, default=1)
    parser.add_argument("--optimizer", type=str, default='adam', choices=['adam', 'adamw'], help="optimizer")
    parser.add_argument('--warmup', type=int, default=0)
    parser.add_argument('--seed', type=int, default=0)
    parser.add_argument('--exp_name', type=str, default='')
    args = parser.parse_args()
    print(args)

    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    torch.cuda.manual_seed(args.seed)

    para_dic = {'': args.model, 'nl': args.num_layers, 'es': args.ego_size, 'hs': args.hidden_size,
                'hd': args.hidden_dropout, 'bs': args.batch_size, 'op': args.optimizer, 
                'lr': args.lr, 'wd': args.weight_decay, 'ls': args.lr_scale, 'bn': args.batch_norm,
                'rs': args.residual, 'sd': args.seed}
    para_dic['warm'] = args.warmup
    exp_name = get_exp_name(args.dataset, para_dic, args.exp_name)

    wandb_name = exp_name.replace('_sd'+str(args.seed), '')
    wandb.init(name=wandb_name, project="xfmr4gl")
    wandb.config.update(args)

    device = f'cuda:{args.device}' if torch.cuda.is_available() else 'cpu'
    device = torch.device(device)

    dataset = DglLinkPropPredDataset(name=f'ogbl-{args.dataset}')
    split_edge = dataset.get_edge_split()

    ego_graphs = np.load(f'data/{args.dataset}-ego-graphs-{args.ego_size}.npy', allow_pickle=True)
    ego_graphs = [torch.LongTensor(ego) for ego in ego_graphs]

    data = dataset[0]
    graph = dgl.remove_self_loop(data)
    graph = dgl.add_self_loop(graph)
    data = graph
    assert args.dataset == "collab"

    train_dataset = LinkPredictionDataset(data, ego_graphs, split_edge['train'])
    train_loader = DataLoader(train_dataset, batch_size=args.batch_size, shuffle=True, num_workers=args.num_workers, collate_fn=make_batcher(), pin_memory=True)

    split_edge['train_eval'] = {
        'edge': split_edge['train']['edge'][:split_edge['valid']['edge'].shape[0]],
        'edge_neg': torch.randint(0, len(ego_graphs), split_edge['valid']['edge'].shape, dtype=torch.long),
    }
    train_eval_dataset = LinkPredictionTestDataset(data, ego_graphs, split_edge['train_eval'])
    train_eval_loader = DataLoader(train_eval_dataset, batch_size=args.eval_batch_size, shuffle=False, num_workers=args.num_workers, collate_fn=make_batcher(), pin_memory=True)

    valid_dataset = LinkPredictionTestDataset(data, ego_graphs, split_edge['valid'])
    valid_loader = DataLoader(valid_dataset, batch_size=args.eval_batch_size, shuffle=False, num_workers=args.num_workers, collate_fn=make_batcher(), pin_memory=True)

    test_dataset = LinkPredictionTestDataset(data, ego_graphs, split_edge['test'])
    test_loader = DataLoader(test_dataset, batch_size=args.eval_batch_size, shuffle=False, num_workers=args.num_workers, collate_fn=make_batcher(), pin_memory=True)

    model = GNNModel(conv_type=args.model, input_size=graph.ndata['feat'].shape[1], hidden_size=args.hidden_size, num_layers=args.num_layers, 
                     num_classes=2, batch_norm=args.batch_norm, residual=args.residual, 
                     dropout=args.hidden_dropout, linear_layer=None, norm=args.norm).to(device)
    predictor = LinkPredictor(args.hidden_size, args.hidden_size, 1,
                              args.num_layers, args.hidden_dropout).to(device)
    if torch.cuda.device_count() > 1:
        print("Let's use", torch.cuda.device_count(), "GPUs!")
        # dim = 0 [30, xxx] -> [10, ...], [10, ...], [10, ...] on 3 GPUs
        model = nn.DataParallel(model)
        predictor = nn.DataParallel(predictor)

    wandb.watch(model, log='all')

    pytorch_total_params = sum(p.numel() for p in model.parameters() if p.requires_grad)
    print('model parameters:', pytorch_total_params)

    evaluator = Evaluator(name=f'ogbl-{args.dataset}')

    try:
        writer = SummaryWriter('runs/' + exp_name)
        hahaha
    except Exception as e:
        print(e)
        writer = None

    for run in range(args.runs):
        if torch.cuda.device_count() > 1:
            model.module.reset_parameters()
            # predictor.reset_parameters()
            predictor.module.reset_parameters()
        else:
            model.reset_parameters()
            predictor.reset_parameters()
        best_val_hits = 0
        cor_test_hits = 0
        patience = 0

        if args.optimizer == 'adam':
            optimizer = torch.optim.Adam(list(model.parameters()) + list(predictor.parameters()), lr=args.lr, weight_decay=args.weight_decay)
        elif args.optimizer == 'adamw':
            optimizer = torch.optim.AdamW(list(model.parameters()) + list(predictor.parameters()), lr=args.lr, weight_decay=args.weight_decay)
        else:
            raise NotImplementedError
        # optimizer = ScheduledOptim(optimizer, args.hidden_size, n_warmup_steps=args.warmup, init_lr_scale=args.lr_scale)

        for epoch in range(1, 1 + args.epochs):
            train_iter = train(model, predictor, train_loader, args, optimizer, device)
            for loss in train_iter:
                results = test(model, predictor, train_eval_loader, evaluator, device)
                print(results)
                results = test(model, predictor, valid_loader, evaluator, device)
                if epoch % args.log_steps == 0:
                    for key, result in results.items():
                        valid_hits = result
                        if writer is not None:
                            writer.add_scalar('loss', loss, epoch)
                            # writer.add_scalar('acc/train', train_hits, epoch)
                            writer.add_scalar('acc/valid', valid_hits, epoch)
                            writer.add_scalar('acc/best_val', best_val_hits, epoch)
                            writer.add_scalar('acc/cor_test', cor_test_hits, epoch)
                            writer.add_scalar('lr', get_lr(optimizer), epoch)

                        wandb.log({'Train Loss': loss, #f'Train {key}': train_hits,
                                f'Valid {key}': valid_hits,
                                f'best_val_hits': best_val_hits, f'cor_test_hits': cor_test_hits,
                                'LR': get_lr(optimizer)})
                        print(f'Run: {run + 1:02d}, '
                            f'Epoch: {epoch:02d}, '
                            f'Loss: {loss:.4f}, '
                            # f'Train {key}: {100 * train_hits:.2f}%, '
                            f'Valid {key}: {100 * valid_hits:.2f}% ')
                        best_key = "Hits@50"
                        if args.dataset == "ppa":
                            best_key = "Hits@100"
                        elif args.dataset == "ddi":
                            best_key = "Hits@20"
                        elif args.dataset == "citation":
                            best_key = "MRR"
                        if key == best_key:
                            if valid_hits > best_val_hits:
                                best_val_hits = valid_hits
                                test_results = test(model, predictor, test_loader, evaluator, device)
                                cor_test_hits = test_results[best_key]
                                patience = 0
                                if torch.cuda.device_count() > 1:
                                    torch.save(model.module.state_dict(), 'saved/' + exp_name + '_model.pt')
                                    torch.save(predictor.module.state_dict(), 'saved/' + exp_name + '_predictor.pt')
                                else:
                                    torch.save(model.state_dict(), 'saved/' + exp_name + '_model.pt')
                                    torch.save(predictor.state_dict(), 'saved/' + exp_name + '_predictor.pt')
                                wandb.save('saved/' + exp_name + '.pt')
                            else:
                                patience += 1
                                if patience >= args.early_stopping:
                                    print('Early stopping...')
                                    wandb.log({'Final val': best_val_hits, 'Final test': cor_test_hits})
                                    exit()


if __name__ == "__main__":
    main()
