import sys

def create_latex_table(bin_dict, title):
    latex_str = f"""
\\begin{{center}}
    Accuracy on {title}
    """
    latex_str_end = """
    \\setlength{\\tabcolsep}{0pt}
    \\renewcommand{\\arraystretch}{0}
        \\begin{tabular}[b]{l@{\;\;}*{10}{c}}
        & 1 & 2 & 3 & 4 & 5 & 6 & 7 & 8 & 9 & 10\\\[1ex]
    """
    latex_str += latex_str_end
    # bin_dict is a dictionary with languages as keys and a list of accuracies as values
    bin_dict = dict(sorted(bin_dict.items(), key=lambda x: int(x[0][1:])))
    for language, accuracies in bin_dict.items():

        if language.startswith("L"):
            k = int(language[1:])
        latex_str += f"$\\altplus{{{k}}}$ & "

        for i in sorted(accuracies.keys()):
            accuracy = accuracies[i]
            if accuracy is not None:
                latex_str += f"\\cellgradient{{{accuracy}}} & "
            else:
                latex_str += "0.00 & "
        latex_str = latex_str[:-2] + "\\\\\n"
    latex_str += """
    \\end{tabular}%
\\hspace{0.5\\bsize}\\llap{\\raisebox{-0.3\\cellheight}{%
\\begin{tikzpicture}[x=\\cellwidth,y=\\cellheight,baseline=0pt,line width=\\bsize]
\\draw (0,10) |- (1,9) |- (2,8) |- (3,7) |- (4,6) |- (5,5) |- (6,4) |- (7,3) |- (8,2) |- (9,1) |- (10,0);
\\end{tikzpicture}%
}}
\\end{center}
"""
    return latex_str

if __name__ == "__main__":
    if len(sys.argv) != 2:
        print("Usage: python script_name.py <csv_file_path>")
        sys.exit(1)

    csv_file_path = sys.argv[1]
    try:
        with open(csv_file_path, 'r') as file:
            lines = file.readlines()
    except FileNotFoundError:
        print(f"Error: CSV file not found at {csv_file_path}")
        sys.exit(1)
    except Exception as e:
        print(f"Error reading CSV file: {e}")
        sys.exit(1)

    # Parse the CSV file manually
    header = lines[0].strip().split(",")
    data = [line.strip().split(",") for line in lines[1:]]

    bins = {
        "100_150": (100, 150),
        "150_200": (150, 200),
        "200_250": (200, 250),
        "250_300": (250, 300),
    }

    grouped_data = {}

    for bin_name, (lower, upper) in bins.items():
        grouped_data[bin_name] = {}
        for row in data:
            row_dict = dict(zip(header, row))
            for col_name in header:
                if col_name.startswith("acc_"):
                    bin_range = col_name.split("_")[1] + "_" + col_name.split("_")[2]
                    if bin_range == bin_name:
                        language = row_dict["language"]
                        layer = int(row_dict["layers"])
                        accuracy = float(row_dict[col_name]) if row_dict[col_name] else None

                        if language not in grouped_data[bin_name]:
                            grouped_data[bin_name][language] = {}
                        if layer not in grouped_data[bin_name][language]:
                            grouped_data[bin_name][language][layer] = None

                        grouped_data[bin_name][language][layer] = accuracy

    # Convert the grouped data to a list of lists for LaTeX table generation
    data_list = []
    for bin_name, languages in grouped_data.items():
        for language, layers in languages.items():
            row = []
            for layer in range(3, 13):
                if layer in layers:
                    row.append(layers[layer])
                else:
                    row.append(0)  # Default value for missing layers
            data_list.append(row)

    with open("latex_tables_combined.tex", "w") as f:
        for bin_name, bin_dict in grouped_data.items():
            bin_name = "[" + bin_name.replace("_", ",") + "]"
            latex_table = create_latex_table(bin_dict, bin_name)
            f.write(latex_table)
            # f.write("\n\\newpage\n")
        print("All LaTeX tables written to latex_tables_combined.tex")


    print("Done.")
