import os
import inspect
import argparse
import sys
import traceback

class Config(object):
    _parser = argparse.ArgumentParser()
    _commands = _parser.add_subparsers(title='components')

    def __init__(self, args, context):
        """Build config by args and context.

        Parameters
        ----------
        args: argparse.Namespace
            Commandline arguments.

        context: argparse.Namespace
            Global context shared by all components.
        """
        self.config_dict = {}
        self.update_config_from_args(args)

    @classmethod
    def define_parser(cls, parser):
        """Define argparse parser.

        Parameters
        ----------
        parser: argparse.ArgumentParser
            The parser.
        """
        pass

    @staticmethod
    def build(args, context):
        """Callback after argparsing, build config and maintain context here.

        Parameters
        ----------
        args: argparse.Namespace
            Commandline arguments.

        context: argparse.Namespace
            Global context shared by all components.
        """
        raise NotImplementedError

    @classmethod
    def register(cls, name, help=None, is_fold=False):
        """ Register subcomponent.

        Parameters
        ----------
        name: str
            Command for subcomponent.
        is_fold: bool
            Whether the component is abstract component.
        
        Examples
        --------
        >>> from graph_learning.config import Config
        >>> @Config.register('foo')
        >>> class FooConfig(Config):
        >>>     pass
        >>> @FooConfig.register('bar')
        >>> class BarConfig(FooConfig):
        >>>     pass

        Then command for define 'bar' component is 'foo bar'.
        """
        subparser = cls._commands.add_parser(name, help=help, description=help)
        def register_subcommand(subclass):
            subclass._parser = subparser
            subclass._commands = subparser.add_subparsers(title='subcommands')
            subclass.define_parser(subparser)
            if not is_fold:
                subparser.set_defaults(func=subclass.build)
            return subclass
        return register_subcommand

    def set_when_empty(self, name, v):
        if getattr(self, name, None) is None:
            kv = {name: v}
            self.config_dict.update(kv)
            self.__dict__.update(kv)

    def fetch_or_set(self, arg, context, typ):
        try:
            arg = typ(arg)
        except:
            arg = multi_getattr(context, arg)
        return arg

    def update_config_from_args(self, args):
        for k, v in args.__dict__.items():
            kv = {k: v}
            self.config_dict.update(kv)
            self.__dict__.update(kv)

# def register_parser_action(parser, action):
#     parser.set_defaults(func=action)

def config_dispatch(top_module, config):
    func = top_module.__init__
    signature = inspect.signature(func)
    names = list(signature.parameters.keys())[1:] # exclude self
    parameters = {name: getattr(config, name)
                  for name in names
                  if hasattr(config, name)}
    try:
        obj = top_module(**parameters)
    except TypeError as e:
        raise TypeError(f'{top_module.__name__}: {e}')
    except Exception as e:
        traceback.print_exc()
        exit(1)
    return obj

def multi_getattr(obj, attr):
    attributes = attr.split(".")
    for i in attributes:
        try:
            obj = getattr(obj, i)
        except AttributeError:
            traceback.print_exc()
            print(f'{attr} needed.')
            exit(1)
    return obj

def save_commands(path, splited_commands):
    with open(path, 'w') as f:
        f.write('\n'.join([' '.join(cmd) for cmd in splited_commands]))

def init_context_namespace(base, name, init=None):
    if not hasattr(base, name):
        setattr(base, name, init)
        return True
    return False

def parse_args():
    parser = Config._parser
    commands = Config._commands
    # Divide argv by commands
    split_argv = [[]]
    for c in sys.argv[1:]:
        if c in commands.choices:
            split_argv.append([c])
        else:
            split_argv[-1].append(c)
    # Initialize namespace
    context = argparse.Namespace()
    setattr(context, 'commands', split_argv[1:-1])
    setattr(context, 'global_', argparse.Namespace())
    #for c in commands.choices:
    #    setattr(context, c.replace('-', '_'), None)
    # Parse each command
    for argv in split_argv[1:]:  # Commands
        n = argparse.Namespace()
        if not argv[0].startswith('global_'):
            print(f'Running: [{" ".join(argv)}]')
        comp_args = parser.parse_args(argv, namespace=n)
        comp_args.func(comp_args, context)
    return context
