"""
Utilities for reinforcement learning algorithms.
"""
import os
os.environ["TF_CPP_MIN_LOG_LEVEL"] = "3"  # Suppress TensorFlow warnings

import numpy as np
import tensorflow as tf
from tensorflow.keras import Sequential
from tensorflow.keras.layers import Dense, LSTM
from tensorflow.keras.optimizers import Adam
import logging
from collections import deque

# Setup logging
logging.basicConfig(level=logging.INFO, format='%(asctime)s - %(levelname)s - %(message)s')

# Shared constants
MODEL_PATH = './MODEL/WDS_Q'
ACTION_SPACE = 5
ACTIONS = [[0,0,0,0], [1,0,0,0], [0,1,0,0], [0,0,1,0], [0,0,0,1]]
MAX_SIZE_MEMORY = 1051200
DISCOUNT_FACTOR = 0.99
HIDDEN_NODES = 100
STATE_SPACE = 25
STATE_SIZE = 4
MINIBATCH_SIZE = 36
UPDATE_TARGET = 12000
LEARNING_RATE = 3e-5
EPOCHS = 1
PER = True
ABS_ERROR_UPPER = 1
N_MODELS = 3
TAU = 0.3  # BCQ threshold
ALPHA = 0.6  # PER prioritization factor
BETA_INCREMENT = 0.0001  # PER beta increment
EPSILON = 0.1  # PER small constant to avoid zero priority

def create_model(state_space: int) -> Sequential:
    """Create a model with LSTM and Dense layers."""
    model = Sequential([
        LSTM(HIDDEN_NODES, input_shape=(STATE_SIZE, state_space), return_sequences=False),
        Dense(HIDDEN_NODES, activation='relu', kernel_regularizer=tf.keras.regularizers.l2(1e-6)),
        Dense(HIDDEN_NODES, activation='relu', kernel_regularizer=tf.keras.regularizers.l2(1e-6)),
        Dense(ACTION_SPACE, activation='linear')
    ])
    model.compile(loss=tf.keras.losses.Huber(delta=1.0), optimizer=Adam(learning_rate=LEARNING_RATE, clipnorm=1))
    return model

def feed_memory(replay_memory: deque) -> int:
    """Load transitions from replay_memory.txt into replay memory."""
    try:
        with open("replay_memory.txt", "r") as file:
            replay_memory.extend([eval(line.rstrip()) for line in file if '\x00' not in line])
        size = len(replay_memory)
        logging.info("Loaded %d transitions into replay memory", size)
        return size
    except FileNotFoundError:
        logging.error("replay_memory.txt not found")
        raise
    except Exception as e:
        logging.error("Error loading replay memory: %s", e)
        raise

def save_models(models: list, model_path: str = MODEL_PATH) -> None:
    """Save all models to disk."""
    for m, model in enumerate(models):
        try:
            model.save(os.path.join(model_path, str(m)), save_format='tf')
            logging.info("Model %d saved successfully", m)
        except Exception as e:
            logging.error("Failed to save model %d: %s", m, e)
