# SPDX-FileCopyrightText: Copyright (c) 2025, NVIDIA CORPORATION & AFFILIATES.
# All rights reserved.
# SPDX-License-Identifier: Apache-2.0

import gradio as gr
import pandas as pd
from apscheduler.schedulers.background import BackgroundScheduler

from src.leaderboard_generator import format_leaderboard_for_display, generate_leaderboard_data
from src.textual_content import CITATION_TEXT, INTRO_TEXT, MOTIVATION_TEXT, SUBMISSION_INSTRUCTIONS, TITLE
from src.utils import restart_space

# Generate leaderboard data
print("Generating leaderboard data...")
try:
    open_source_df, closed_df = generate_leaderboard_data()
    # Format for display
    open_source_display = format_leaderboard_for_display(open_source_df)
    closed_display = format_leaderboard_for_display(closed_df)
    print(f"Generated leaderboards: {len(open_source_df)} open source models, {len(closed_df)} closed models")
except Exception as e:
    print(f"Error generating leaderboard data: {e}")
    # Create empty dataframes as fallback
    open_source_display = pd.DataFrame()
    closed_display = pd.DataFrame()

print("Initializing leaderboard...")
demo = gr.Blocks(theme=gr.themes.Default(primary_hue="green", secondary_hue="green"))
with demo:

    gr.HTML(TITLE)
    gr.Markdown(INTRO_TEXT)
    gr.Markdown(MOTIVATION_TEXT)

    with gr.Tabs(elem_classes="tab-buttons") as tabs:

        #### Main Leaderboard ####
        with gr.TabItem("🏅 Complete Judge Performance Matrix"):
            gr.Markdown(
                """
            ## Complete Judge Performance Matrix
            
            This leaderboard ranks LLM judges based on their agreement with human annotators across multiple metrics:
            
            - **Correlation (r)**: Pearson correlation between judge scores and human average scores
            - **Cohen's Kappa (κ)**: Inter-rater reliability measure accounting for chance agreement
            - **Z-Score**: Standardized measure of how the judge's agreement compares to human annotators
            - **Human-Like**: Whether the judge performs within human-like range (|z| < 1)
            
            **Legend**:
            - ✅ **Yes**: Model performs within human-like range (|z| < 1)
            - ⚙️ **Super-Consistent**: Model exceeds human agreement by >1 standard deviation (z > 1)
            - ❌ **No**: Model underperforms compared to humans (z < -1)
            """
            )

            with gr.Column():
                # Open Source Models Leaderboard
                gr.Markdown("### Open Source Models Leaderboard")
                gr.Markdown("*Rankings are determined by Z-Score values (higher is better)*")

                open_source_leaderboard = gr.Dataframe(
                    value=open_source_display,
                    interactive=False,
                    wrap=True,
                    elem_classes="custom-df",
                    row_count=(20, "dynamic"),
                    col_count=(7, "fixed"),
                    headers=["Rank", "Judge", "Z-Score", "Correlation (r)", "Cohen's Kappa (κ)", "|z|", "Human-Like?"],
                )

                # Closed Models Leaderboard
                gr.Markdown("### Closed Models Leaderboard")
                gr.Markdown("*Rankings are determined by Z-Score values (higher is better)*")

                closed_leaderboard = gr.Dataframe(
                    value=closed_display,
                    interactive=False,
                    wrap=True,
                    elem_classes="custom-df",
                    row_count=(10, "dynamic"),
                    col_count=(7, "fixed"),
                    headers=["Rank", "Judge", "Z-Score", "Correlation (r)", "Cohen's Kappa (κ)", "|z|", "Human-Like?"],
                )

                # Refresh button
                with gr.Row():
                    refresh_button = gr.Button("🔄 Refresh Leaderboards", variant="secondary")

                def refresh_leaderboards():
                    try:
                        open_df, closed_df = generate_leaderboard_data()
                        open_display = format_leaderboard_for_display(open_df)
                        closed_display = format_leaderboard_for_display(closed_df)
                        return open_display, closed_display
                    except Exception as e:
                        print(f"Error refreshing leaderboards: {e}")
                        return open_source_display, closed_display

                refresh_button.click(fn=refresh_leaderboards, outputs=[open_source_leaderboard, closed_leaderboard])

        #### Submission instructions ####
        with gr.TabItem("🚀 Submit here!"):
            with gr.Column():
                gr.Markdown(SUBMISSION_INSTRUCTIONS)

        #### Citation ####
        with gr.TabItem("📙 Citation"):
            with gr.Column():
                gr.Markdown(CITATION_TEXT)


# Launch the app
scheduler = BackgroundScheduler()
scheduler.add_job(restart_space, "interval", hours=12)
scheduler.start()
demo.queue(default_concurrency_limit=40).launch(ssr_mode=False)
