#!/usr/bin/env python3
# SPDX-FileCopyrightText: Copyright (c) 2025, NVIDIA CORPORATION & AFFILIATES.
# All rights reserved.
# SPDX-License-Identifier: Apache-2.0
"""
Export leaderboard data to CSV files.

This script generates the open source and closed model leaderboards and exports them to CSV files.
"""

import argparse
import os
import sys
from datetime import datetime

# Add the parent directory to the Python path to import from src
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from src.leaderboard_generator import generate_leaderboard_data


def main():
    parser = argparse.ArgumentParser(description="Export leaderboard data to CSV files")
    parser.add_argument(
        "--analysis-dir",
        default="./benchmark/judge_results/",
        help="Directory containing judge results (default: ./benchmark/judge_results/)",
    )
    parser.add_argument(
        "--output-dir",
        default="./benchmark/",
        help="Directory to save the CSV files (default: ./benchmark/)",
    )
    parser.add_argument(
        "--timestamp",
        action="store_true",
        help="Add timestamp to the output filenames",
    )

    args = parser.parse_args()

    # Ensure output directory exists
    os.makedirs(args.output_dir, exist_ok=True)

    print("Loading annotations from HuggingFace dataset...")
    print("Make sure to set the 'access_token_for_judges_verdict_private' environment variable")
    print(f"Loading judge results from: {args.analysis_dir}")
    print(f"Output directory: {args.output_dir}")

    # Generate leaderboard data
    print("\nGenerating leaderboard data...")
    open_source_df, closed_df = generate_leaderboard_data(analysis_dir=args.analysis_dir)

    # Create output filenames
    if args.timestamp:
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        open_source_filename = f"leaderboard_open_source_{timestamp}.csv"
        closed_filename = f"leaderboard_closed_{timestamp}.csv"
    else:
        open_source_filename = "leaderboard_open_source.csv"
        closed_filename = "leaderboard_closed.csv"

    open_source_path = os.path.join(args.output_dir, open_source_filename)
    closed_path = os.path.join(args.output_dir, closed_filename)

    # Save DataFrames to CSV
    if not open_source_df.empty:
        open_source_df.to_csv(open_source_path, index=False)
        print(f"\nOpen source leaderboard saved to: {open_source_path}")
        print(f"  - Number of models: {len(open_source_df)}")
        print(f"  - Columns: {', '.join(open_source_df.columns)}")
    else:
        print("\nNo open source models found in the leaderboard")

    if not closed_df.empty:
        closed_df.to_csv(closed_path, index=False)
        print(f"\nClosed model leaderboard saved to: {closed_path}")
        print(f"  - Number of models: {len(closed_df)}")
        print(f"  - Columns: {', '.join(closed_df.columns)}")
    else:
        print("\nNo closed models found in the leaderboard")

    # Print summary statistics
    print("\n=== Summary ===")
    if not open_source_df.empty:
        print("\nOpen Source Models:")
        print(f"  Top model: {open_source_df.iloc[0]['Judge']} (Z-Score: {open_source_df.iloc[0]['Z-Score']})")
        print(f"  Average Z-Score: {open_source_df['Z-Score'].mean():.3f}")
        print(f"  Human-like models: {(open_source_df['Human-Like?'] == '✅ Yes').sum()}")
        print(f"  Super-consistent models: {(open_source_df['Human-Like?'] == '⚙️ Super-Consistent').sum()}")

    if not closed_df.empty:
        print("\nClosed Models:")
        print(f"  Top model: {closed_df.iloc[0]['Judge']} (Z-Score: {closed_df.iloc[0]['Z-Score']})")
        print(f"  Average Z-Score: {closed_df['Z-Score'].mean():.3f}")
        print(f"  Human-like models: {(closed_df['Human-Like?'] == '✅ Yes').sum()}")
        print(f"  Super-consistent models: {(closed_df['Human-Like?'] == '⚙️ Super-Consistent').sum()}")


if __name__ == "__main__":
    main()
