# SPDX-FileCopyrightText: Copyright (c) 2025, NVIDIA CORPORATION & AFFILIATES.
# All rights reserved.
# SPDX-License-Identifier: Apache-2.0

"""
Judge configuration module - uses YAML-based configuration exclusively.

This module provides access to judge configurations defined in YAML files.
By default, it uses config/judge_config_litellm.yaml. To use a different config file,
set the JUDGE_CONFIG_PATH environment variable.
"""

import os
from typing import Optional

from .judge_config_manager import JudgeConfigManager, JudgeModelConfig

# Load the configuration manager
# Check for environment variable to override default config path
# Default to LiteLLM config if no env var is set
config_path = os.environ.get("JUDGE_CONFIG_PATH", "config/judge_config_litellm.yaml")
_config_manager = JudgeConfigManager(config_path)


def get_judge_model_config(model_name: str) -> Optional[JudgeModelConfig]:
    """Get detailed configuration for a specific judge model."""
    return _config_manager.get_model(model_name)
