#!/usr/bin/env python3
# SPDX-FileCopyrightText: Copyright (c) 2025, NVIDIA CORPORATION & AFFILIATES.
# All rights reserved.
# SPDX-License-Identifier: Apache-2.0
"""
Generate leaderboard data for LLM judges by computing correlation, Cohen's Kappa, and Z-scores.
"""

import os
from typing import Any, Dict, List, Optional, Tuple

import numpy as np
import pandas as pd
from datasets import load_dataset
from sklearn.metrics import cohen_kappa_score

from src.judge_config_manager import JudgeConfigManager
from src.judge_loader import get_available_judges, load_and_aggregate_judge_scores
from src.llm_score_utils import convert_to_3_point_scale


def calculate_cohens_kappa(rater1_scores: List[float], rater2_scores: List[float]) -> float:
    """Calculate Cohen's Kappa between two raters using sklearn with quadratic weights."""
    # Filter out pairs where either rater has None
    valid_pairs = []
    for r1, r2 in zip(rater1_scores, rater2_scores):
        if r1 is not None and r2 is not None:
            c1 = convert_to_3_point_scale(r1)
            c2 = convert_to_3_point_scale(r2)
            if c1 is not None and c2 is not None:
                valid_pairs.append((c1, c2))

    if not valid_pairs:
        return 0.0

    # Convert to string labels to avoid sklearn treating them as continuous
    rater1_valid = [str(pair[0]) for pair in valid_pairs]
    rater2_valid = [str(pair[1]) for pair in valid_pairs]

    # Use sklearn's cohen_kappa_score with quadratic weights
    return cohen_kappa_score(rater1_valid, rater2_valid, weights="quadratic", labels=["0.0", "0.5", "1.0"])


def load_human_annotations() -> Dict[str, Dict]:
    """Load human annotations from HuggingFace dataset.

    Returns:
        Dictionary mapping (question, gt_answer, gen_answer) to annotation data
    """
    # Get HuggingFace token from environment
    hf_token = os.environ.get("access_token_for_judges_verdict_private")
    if not hf_token:
        raise ValueError(
            "HuggingFace token not found. Please set the 'access_token_for_judges_verdict_private' environment variable."
        )

    # Load dataset from HuggingFace
    print("Loading dataset from HuggingFace...")
    dataset = load_dataset("nvidia/judges-verdict-private", split="train", token=hf_token)

    # Convert to list of dicts
    data = [item for item in dataset]

    # Create mapping from (question, gt_answer, gen_answer) to annotations
    annotations_map = {}

    for item in data:
        # Create unique key for this item
        key = (item["question"], item["gt_answer"], item["gen_answer"])

        # Extract human scores
        sorted_annotations = sorted(item["annotations"], key=lambda x: x["annotator"])

        human_scores = []
        for annotation in sorted_annotations:
            score = annotation["score"]
            # Convert to float, handle empty/None as None
            if score and str(score).strip():
                human_scores.append(float(score))
            else:
                human_scores.append(None)

        annotations_map[key] = {
            "dataset_name": item["dataset_name"],
            "item_name": item["item_name"],
            "human_scores": human_scores,
            "annotators": [ann["annotator"] for ann in sorted_annotations],
        }

    return annotations_map


def calculate_correlation(judge_scores: List[float], human_avg_scores: List[float]) -> float:
    """Calculate Pearson correlation between judge and human average scores."""
    # Filter out pairs where either score is None
    valid_pairs = []
    for j, h in zip(judge_scores, human_avg_scores):
        if j is not None and h is not None:
            valid_pairs.append((j, h))

    if len(valid_pairs) < 2:
        return 0.0

    judge_valid = [pair[0] for pair in valid_pairs]
    human_valid = [pair[1] for pair in valid_pairs]

    # Calculate Pearson correlation
    return np.corrcoef(judge_valid, human_valid)[0, 1]


def compute_judge_metrics(
    judge_name: str, human_annotations: Dict[str, Dict], analysis_dir: str
) -> Optional[Dict[str, Any]]:
    """Compute all metrics for a single judge."""

    # Load judge scores
    judge_scores = load_and_aggregate_judge_scores(
        judge_name, analysis_dir, convert_to_3_point=True, convert_first=True
    )

    if not judge_scores:
        return None

    # Prepare data for analysis
    judge_score_list = []
    human_avg_list = []
    human_scores_by_annotator = {"human1": [], "human2": [], "human3": []}
    all_scores_by_rater = {"Human 1": [], "Human 2": [], "Human 3": [], judge_name: []}

    matched_items = 0

    for key, human_data in human_annotations.items():
        if key in judge_scores:
            human_scores = human_data["human_scores"]
            judge_score = judge_scores[key]

            # Skip if any human score is None/empty or judge score is None
            if None in human_scores or judge_score is None:
                continue

            matched_items += 1

            # For correlation calculation
            human_avg = np.mean(human_scores)
            judge_score_list.append(judge_score)
            human_avg_list.append(human_avg)

            # For Cohen's Kappa calculation
            for i, human_score in enumerate(human_scores[:3]):
                human_scores_by_annotator[f"human{i+1}"].append((human_score, judge_score))

            # For outlier analysis
            all_scores_by_rater["Human 1"].append(human_scores[0])
            all_scores_by_rater["Human 2"].append(human_scores[1])
            all_scores_by_rater["Human 3"].append(human_scores[2])
            all_scores_by_rater[judge_name].append(judge_score)

    if matched_items == 0:
        return None

    # Calculate correlation
    correlation = calculate_correlation(judge_score_list, human_avg_list)

    # Calculate Cohen's Kappa with each human
    cohens_kappas = {}
    for i in range(3):
        human_scores = [pair[0] for pair in human_scores_by_annotator[f"human{i+1}"]]
        judge_scores_for_human = [pair[1] for pair in human_scores_by_annotator[f"human{i+1}"]]
        cohens_kappas[f"human{i+1}"] = calculate_cohens_kappa(human_scores, judge_scores_for_human)

    avg_cohens_kappa = np.mean(list(cohens_kappas.values()))

    # Calculate Z-score for outlier analysis
    rater_names = ["Human 1", "Human 2", "Human 3", judge_name]

    # Calculate pairwise Cohen's Kappa between all pairs
    avg_kappas_by_rater = {}

    for i, rater1 in enumerate(rater_names):
        kappas_with_others = []

        for j, rater2 in enumerate(rater_names):
            if i < j:  # Only calculate once for each pair
                # Calculate Cohen's Kappa for this pair across all items
                kappa = calculate_cohens_kappa(all_scores_by_rater[rater1], all_scores_by_rater[rater2])
                kappas_with_others.append(kappa)

                if rater2 not in avg_kappas_by_rater:
                    avg_kappas_by_rater[rater2] = []
                avg_kappas_by_rater[rater2].append(kappa)

        if rater1 not in avg_kappas_by_rater:
            avg_kappas_by_rater[rater1] = []
        avg_kappas_by_rater[rater1].extend(kappas_with_others)

    # Calculate average kappa for each rater
    avg_kappas = {}
    for rater, kappas in avg_kappas_by_rater.items():
        avg_kappas[rater] = np.mean(kappas) if kappas else 0

    # Calculate z-score
    judge_avg_kappa = avg_kappas[judge_name]
    human_avg_kappas = [avg_kappas[r] for r in ["Human 1", "Human 2", "Human 3"]]
    human_mean_kappa = np.mean(human_avg_kappas)
    human_std_kappa = np.std(human_avg_kappas)

    if human_std_kappa > 0:
        z_score = (judge_avg_kappa - human_mean_kappa) / human_std_kappa
    else:
        z_score = 0

    return {
        "judge_name": judge_name,
        "correlation": correlation,
        "cohens_kappa": avg_cohens_kappa,
        "z_score": z_score,
        "matched_items": matched_items,
    }


def get_human_like_status(z_score: float) -> str:
    """Determine human-like status based on z-score."""
    abs_z = abs(z_score)
    if abs_z < 1:
        return "✅ Yes"
    elif z_score > 1:
        return "⚙️ Super-Consistent"
    else:  # z_score < -1
        return "❌ No"


def generate_leaderboard_data(analysis_dir: str = "./benchmark/judge_results/") -> Tuple[pd.DataFrame, pd.DataFrame]:
    """Generate leaderboard data for open source and closed models.

    Args:
        analysis_dir: Directory containing judge results

    Returns:
        Tuple of (open_source_df, closed_df) DataFrames
    """
    # Load judge configurations to determine which models are closed source
    config_manager = JudgeConfigManager()

    # Try loading different config files - we'll use whichever is available
    config_paths = [
        "./config/judge_config_litellm.yaml",
        "./config/judge_config_local.yaml",
        "./config/judge_config.yaml",
    ]
    config_loaded = False

    for config_path in config_paths:
        try:
            config_manager.load_config(config_path)
            config_loaded = True
            break
        except FileNotFoundError:
            continue

    if not config_loaded:
        print("Warning: Could not load judge configuration file. All models will be treated as open source.")
        config_manager = None

    # Load human annotations
    print("Loading human annotations...")
    try:
        human_annotations = load_human_annotations()
        print(f"Loaded {len(human_annotations)} annotated items")
    except Exception as e:
        print(f"Error loading annotations: {e}")
        return pd.DataFrame(), pd.DataFrame()

    # Get available judges
    try:
        judges = get_available_judges(analysis_dir)
    except Exception as e:
        print(f"Error getting judge models: {e}")
        return pd.DataFrame(), pd.DataFrame()

    if not judges:
        print(f"No judge models found in {analysis_dir}")
        return pd.DataFrame(), pd.DataFrame()

    # Compute metrics for all judges
    all_results = []

    for judge in judges:
        print(f"Processing {judge}...")

        try:
            metrics = compute_judge_metrics(judge, human_annotations, analysis_dir)

            if metrics:
                all_results.append(metrics)
        except Exception as e:
            print(f"Error processing {judge}: {e}")
            continue

    if not all_results:
        print("No results generated for any judge")
        return pd.DataFrame(), pd.DataFrame()

    # Separate open source and closed models
    open_source_results = []
    closed_results = []

    for result in all_results:
        judge_name = result["judge_name"]

        # Check if it's a closed model from config
        is_closed = False
        if config_manager:
            model_config = config_manager.get_model(judge_name)
            if model_config:
                is_closed = model_config.is_closed

        # Create display name - prettify the judge name
        display_name = judge_name.replace("nvdev_", "").replace("local_", "").replace("_", "/")

        row_data = {
            "Judge": display_name,
            "Correlation (r)": round(result["correlation"], 3),
            "Cohen's Kappa (κ)": round(result["cohens_kappa"], 3),
            "Z-Score": round(result["z_score"], 2),
            "|z|": round(abs(result["z_score"]), 2),
            "Human-Like?": get_human_like_status(result["z_score"]),
        }

        if is_closed:
            closed_results.append(row_data)
        else:
            open_source_results.append(row_data)

    # Create DataFrames and sort by correlation
    open_source_df = pd.DataFrame(open_source_results)
    if not open_source_df.empty:
        open_source_df = open_source_df.sort_values("Z-Score", ascending=False)
        open_source_df.insert(0, "Rank", range(1, len(open_source_df) + 1))
        # Reorder columns - Z-Score first after Judge
        cols = ["Rank", "Judge", "Z-Score", "Correlation (r)", "Cohen's Kappa (κ)", "|z|", "Human-Like?"]
        open_source_df = open_source_df[cols]

    closed_df = pd.DataFrame(closed_results)
    if not closed_df.empty:
        closed_df = closed_df.sort_values("Z-Score", ascending=False)
        closed_df.insert(0, "Rank", range(1, len(closed_df) + 1))
        # Reorder columns - Z-Score first after Judge
        cols = ["Rank", "Judge", "Z-Score", "Correlation (r)", "Cohen's Kappa (κ)", "|z|", "Human-Like?"]
        closed_df = closed_df[cols]

    return open_source_df, closed_df


def format_leaderboard_for_display(df: pd.DataFrame) -> pd.DataFrame:
    """Format the leaderboard dataframe for better display in Gradio."""
    # Return the dataframe as-is to preserve numeric types for proper sorting
    # Gradio will handle the display formatting
    return df


if __name__ == "__main__":
    # Test the leaderboard generation
    open_source_lb, closed_lb = generate_leaderboard_data()

    print("\nOpen Source Models Leaderboard:")
    print(open_source_lb)

    print("\nClosed Models Leaderboard:")
    print(closed_lb)
