# SPDX-FileCopyrightText: Copyright (c) 2025, NVIDIA CORPORATION & AFFILIATES.
# All rights reserved.
# SPDX-License-Identifier: Apache-2.0

import os

from gradio.themes.utils import colors
from huggingface_hub import HfApi

### General settings ###
LINKS_COLOR = colors.green.c500
TOKEN = os.environ.get("HF_TOKEN")
OWNER = "NVIDIA"
REPO_ID = f"{OWNER}/judges-verdict-leaderboard"

LOCAL_RESULTS_DIR = "./benchmark/"  # local dir to store results

API = HfApi(token=TOKEN)

### Leaderboard table settings ###
LB_ALLOWED_MODELS = None  # Show all judge models
LB_DEFAULT_MODELS = [
    "gpt-4o",
    "gpt-4o-mini",
    "claude-3-5-sonnet-latest",
    "gemini-1.5-pro",
    "gemini-1.5-flash",
]  # Default judge models to show

LB_ALLOWED_DATASETS = [
    "coral",
    "dc767",
    "enterprise_rag_benchmark",
    "financebench_10k",
    "hotpotqa",
    "ragbattlepacket",
    "squad",
    "techqa",
]  # Datasets in Judge's Verdict benchmark

LB_DEFAULT_COLUMNS = [
    "judge",
    "framework",
    "overall_correlation",
    "overall_kappa",
    "avg_human_score",
    "avg_judge_score",
    "n_samples",
]  # columns to show in the leaderboard table

LB_HIDE_COLUMNS = ["filename", "config_path"]  # columns to hide
LB_MARKDOWN_COLUMNS = ["judge", "dataset"]  # columns to show as markdown
LB_HTML_COLUMNS = ["judge"]  # columns to show as html

### Mapping from judge name to pretty judge name ###
METHOD_TO_PRETTY_NAME = {
    # OpenAI models
    "gpt-4o": "GPT-4o",
    "gpt-4o-mini": "GPT-4o-mini",
    "gpt-4-turbo": "GPT-4-Turbo",
    "gpt-3.5-turbo": "GPT-3.5-Turbo",
    # Anthropic models
    "claude-3-5-sonnet-latest": "Claude 3.5 Sonnet",
    "claude-3-opus": "Claude 3 Opus",
    "claude-3-sonnet": "Claude 3 Sonnet",
    "claude-3-haiku": "Claude 3 Haiku",
    # Google models
    "gemini-1.5-pro": "Gemini 1.5 Pro",
    "gemini-1.5-flash": "Gemini 1.5 Flash",
    "gemini-pro": "Gemini Pro",
    # Open models
    "meta-llama/Meta-Llama-3.1-70B-Instruct": "Llama 3.1 70B",
    "meta-llama/Meta-Llama-3.1-8B-Instruct": "Llama 3.1 8B",
    "Qwen/Qwen2.5-72B-Instruct": "Qwen2.5 72B",
    "Qwen/Qwen2.5-7B-Instruct": "Qwen2.5 7B",
    "mistralai/Mixtral-8x7B-Instruct-v0.1": "Mixtral 8x7B",
    # NVIDIA models
    "nvidia/nemotron-4-340b-instruct": "Nemotron-4 340B",
    # Local models
    "local/qwen-0.5b": "Qwen 0.5B (Local)",
}

### Mapping from pretty judge name to additional info ###
PRETTY_NAME_TO_PAPER_LINK = {
    # OpenAI models
    "GPT-4o": f"GPT-4o (<a href='https://openai.com/research/gpt-4' style='color: {LINKS_COLOR};'>model card</a>)",
    "GPT-4o-mini": f"GPT-4o-mini (<a href='https://openai.com/blog/gpt-4o-mini' style='color: {LINKS_COLOR};'>announcement</a>)",
    "GPT-4-Turbo": f"GPT-4-Turbo (<a href='https://openai.com/research/gpt-4' style='color: {LINKS_COLOR};'>model card</a>)",
    "GPT-3.5-Turbo": f"GPT-3.5-Turbo (<a href='https://platform.openai.com/docs/models/gpt-3-5' style='color: {LINKS_COLOR};'>documentation</a>)",
    # Anthropic models
    "Claude 3.5 Sonnet": f"Claude 3.5 Sonnet (<a href='https://www.anthropic.com/claude' style='color: {LINKS_COLOR};'>model card</a>)",
    "Claude 3 Opus": f"Claude 3 Opus (<a href='https://www.anthropic.com/claude' style='color: {LINKS_COLOR};'>model card</a>)",
    "Claude 3 Sonnet": f"Claude 3 Sonnet (<a href='https://www.anthropic.com/claude' style='color: {LINKS_COLOR};'>model card</a>)",
    "Claude 3 Haiku": f"Claude 3 Haiku (<a href='https://www.anthropic.com/claude' style='color: {LINKS_COLOR};'>model card</a>)",
    # Google models
    "Gemini 1.5 Pro": f"Gemini 1.5 Pro (<a href='https://deepmind.google/technologies/gemini/' style='color: {LINKS_COLOR};'>model card</a>)",
    "Gemini 1.5 Flash": f"Gemini 1.5 Flash (<a href='https://deepmind.google/technologies/gemini/' style='color: {LINKS_COLOR};'>model card</a>)",
    "Gemini Pro": f"Gemini Pro (<a href='https://deepmind.google/technologies/gemini/' style='color: {LINKS_COLOR};'>model card</a>)",
    # Open models
    "Llama 3.1 70B": f"Llama 3.1 70B (<a href='https://huggingface.co/meta-llama/Meta-Llama-3.1-70B-Instruct' style='color: {LINKS_COLOR};'>model card</a>)",
    "Llama 3.1 8B": f"Llama 3.1 8B (<a href='https://huggingface.co/meta-llama/Meta-Llama-3.1-8B-Instruct' style='color: {LINKS_COLOR};'>model card</a>)",
    "Qwen2.5 72B": f"Qwen2.5 72B (<a href='https://huggingface.co/Qwen/Qwen2.5-72B-Instruct' style='color: {LINKS_COLOR};'>model card</a>)",
    "Qwen2.5 7B": f"Qwen2.5 7B (<a href='https://huggingface.co/Qwen/Qwen2.5-7B-Instruct' style='color: {LINKS_COLOR};'>model card</a>)",
    "Mixtral 8x7B": f"Mixtral 8x7B (<a href='https://huggingface.co/mistralai/Mixtral-8x7B-Instruct-v0.1' style='color: {LINKS_COLOR};'>model card</a>)",
    # NVIDIA models
    "Nemotron-4 340B": f"Nemotron-4 340B (<a href='https://huggingface.co/nvidia/nemotron-4-340b-instruct' style='color: {LINKS_COLOR};'>model card</a>)",
    # Local models
    "Qwen 0.5B (Local)": f"Qwen 0.5B Local (<a href='https://huggingface.co/Qwen/Qwen2-0.5B-Instruct' style='color: {LINKS_COLOR};'>model card</a>)",
}

PRETTY_NAME_TO_ADDITIONAL_INFO = {k: "" for k, _ in PRETTY_NAME_TO_PAPER_LINK.items()}

### Mapping from dataset name to dataset info ###
DATASET_PAPER_LINK = {
    "coral": f"[CORAL](<a href='https://huggingface.co/datasets/ariya2357/CORAL' style='color: {LINKS_COLOR};'>dataset</a>)",
    "dc767": f"[DC767](<a href='https://github.com/jestimmer/dc767' style='color: {LINKS_COLOR};'>dataset</a>)",
    "enterprise_rag_benchmark": f"[Enterprise RAG](<a href='https://huggingface.co/datasets/nvidia/enterprise-rag-benchmark' style='color: {LINKS_COLOR};'>dataset</a>)",
    "financebench_10k": f"[FinanceBench](<a href='https://huggingface.co/datasets/PatronusAI/financebench' style='color: {LINKS_COLOR};'>dataset</a>)",
    "hotpotqa": f"[HotpotQA](<a href='https://hotpotqa.github.io/' style='color: {LINKS_COLOR};'>dataset</a>)",
    "ragbattlepacket": f"[RAG Battle](<a href='https://huggingface.co/datasets/ragbattlepacket/rag-battle' style='color: {LINKS_COLOR};'>dataset</a>)",
    "squad": f"[SQuAD](<a href='https://rajpurkar.github.io/SQuAD-explorer/' style='color: {LINKS_COLOR};'>dataset</a>)",
    "techqa": f"[TechQA](<a href='https://huggingface.co/datasets/PragmaticMachineLearning/TechQA' style='color: {LINKS_COLOR};'>dataset</a>)",
}
