# SPDX-FileCopyrightText: Copyright (c) 2025, NVIDIA CORPORATION & AFFILIATES.
# All rights reserved.
# SPDX-License-Identifier: Apache-2.0

"""
Judge configuration module - uses YAML-based configuration exclusively.

This module provides access to judge configurations defined in YAML files.
All configuration must be done through YAML configuration files in the config/ folder.
"""

from typing import List, Dict
from .judge_config_manager import JudgeConfigManager

# Load the configuration manager
_config_manager = JudgeConfigManager()

def get_judge_config_manager() -> JudgeConfigManager:
    """Get the judge configuration manager."""
    return _config_manager

def get_default_judges() -> List[str]:
    """Return the default list of judge models from YAML config."""
    group = _config_manager.get_group('default')
    if group:
        return group.models
    return []

def get_default_judge_workers() -> Dict[str, int]:
    """Return the worker configuration for judge models from YAML config."""
    workers = {}
    for model_name, model_config in _config_manager.models.items():
        workers[model_name] = model_config.num_workers
    return workers

def get_rag_judges() -> List[str]:
    """Return the judge subset used for RAG evaluation from YAML config."""
    group = _config_manager.get_group('rag')
    if group:
        return group.models
    return []

def get_oneapi_judges() -> List[str]:
    """Return the list of OneAPI judge models from YAML config."""
    group = _config_manager.get_group('oneapi')
    if group:
        return group.models
    return []

def get_oneapi_config(model_name: str) -> Dict[str, str]:
    """Get OneAPI configuration for a model from YAML config."""
    model_config = _config_manager.get_model(model_name)
    if model_config and model_config.deployment:
        return {
            "deployment": model_config.deployment,
            "api_version": model_config.api_version or "2024-12-01-preview"
        }
    # Model not found in config
    raise ValueError(f"Model {model_name} not found in configuration. Please add it to your judge configuration YAML file")

def get_oneapi_base_url(model_name: str) -> str:
    """Get the OneAPI base URL for a model from YAML config."""
    model_config = _config_manager.get_model(model_name)
    if model_config:
        base_url = model_config.get_effective_base_url()
        if base_url:
            return base_url
    
    # Model not found or no base URL
    raise ValueError(f"Model {model_name} not found in configuration or has no base URL. Please add it to your judge configuration YAML file")

def get_benchmark_models() -> List[str]:
    """Get the list of models to benchmark from YAML config."""
    group = _config_manager.get_group('benchmark')
    if group:
        return group.models
    return []

def get_judges_from_group(group_name: str) -> List[str]:
    """Get judge models from a configuration group."""
    group = _config_manager.get_group(group_name)
    if group:
        return group.models
    return []

def get_judge_model_config(model_name: str):
    """Get detailed configuration for a specific judge model."""
    return _config_manager.get_model(model_name)