from gym.core import Wrapper
import time


class Monitor(Wrapper):
    def __init__(self, env, allow_early_resets=False,
                 reset_keywords=(), info_keywords=()):
        Wrapper.__init__(self, env=env)
        self.tstart = time.time()
        self.reset_keywords = reset_keywords
        self.info_keywords = info_keywords
        self.allow_early_resets = allow_early_resets
        self.rewards = None
        self.needs_reset = True
        self.episode_rewards = []
        self.episode_lengths = []
        self.episode_times = []
        self.total_steps = 0
        self.current_reset_info = {}

    def reset(self, **kwargs):
        self.reset_state()
        for k in self.reset_keywords:
            v = kwargs.get(k)
            if v is None:
                msg = 'Expected you to pass kwarg {} into reset'.format(k)
                raise ValueError(msg)
            self.current_reset_info[k] = v
        return self.env.reset(**kwargs)

    def reset_state(self):
        if not self.allow_early_resets and not self.needs_reset:
            msg = "Tried to reset an environment before done." \
                  " If you want to allow early resets," \
                  " wrap your env with " \
                  " Monitor(env, path, allow_early_resets=True)"
            raise RuntimeError(msg)
        self.rewards = []
        self.needs_reset = False

    def step(self, action):
        if self.needs_reset:
            msg = "Tried to step environment that needs reset"
            raise RuntimeError(msg)
        ob, rew, done, info = self.env.step(action)
        self.update(ob, rew, done, info)
        return (ob, rew, done, info)

    def update(self, ob, rew, done, info):
        self.rewards.append(rew)
        if done:
            self.needs_reset = True
            eprew = sum(self.rewards)
            eplen = len(self.rewards)
            epinfo = {
                "score": round(eprew, 6),
                "len": eplen,
                "time": round(time.time() - self.tstart, 6)
            }
            for k in self.info_keywords:
                epinfo[k] = info[k]
            self.episode_rewards.append(eprew)
            self.episode_lengths.append(eplen)
            self.episode_times.append(time.time() - self.tstart)
            epinfo.update(self.current_reset_info)
            if isinstance(info, dict):
                info['episode'] = epinfo

        self.total_steps += 1

    def get_total_steps(self):
        return self.total_steps

    def get_episode_rewards(self):
        return self.episode_rewards

    def get_episode_lengths(self):
        return self.episode_lengths

    def get_episode_times(self):
        return self.episode_times
