
import pennylane as qml
from pennylane.ops import CNOT, RX, RY, RZ, CZ
from pennylane import numpy as np
import random
import numpy.linalg as la
import math
from math import pi
from datetime import datetime
import os




def gd_optimzier(ansatz, circuit_name, H, excitations, qubits, theta, noise_gaussian_constant, noise_gaussian_rate, lr, iteration, n_check):

    grad_func = qml.grad(ansatz)
    params = np.copy(theta)
    n_params = len(params)

    gradnorm, loss = np.zeros(iteration), np.zeros(iteration)

    t1 = datetime.now()
    noise_gamma = np.zeros(n_params)
    for j in range(n_params):
        noise_gamma[j] = noise_gaussian_constant + math.sqrt(1/(96*64*n_params))

    for j in range(iteration):
        loss[j] = ansatz(params, obs=H, qubits=qubits, excitations=excitations)
        params.requires_grad = True
        gradient_params = grad_func(params, obs=H, qubits=qubits, excitations=excitations)
        gradnorm[j] = la.norm(gradient_params)
        noise = [np.random.normal(0,noise_gamma[i],1)[0] for i in range(len(params))]
        if(noise_gaussian_rate==2):
            noise_gamma = [0.5*math.sqrt(gradient_params[i]**2) for i in range(len(params))]
        else:
            noise_gamma = [noise_gaussian_constant+noise_gaussian_rate*math.sqrt(gradient_params[i]**2/(96*64*len(params)*loss[j]**2)) for i in range(len(params))]
        gradient_params = gradient_params + noise

        params = params - lr*gradient_params

        if(j%n_check==0):
            t2 = datetime.now()
            print(j, "th", " loss : %.6f" % loss[j], "gradnorm_theta: %.6f" % gradnorm[j], ",", (t2-t1).seconds, "seconds")
            t1 = t2
    return loss, gradnorm, params

def chem_gd_gaussian(ansatz, circuit_name, H, excitations, qubits, n_params, gamma, noise_gaussian_constant, noise_gaussian_rate, lr, iteration, n_time, n_check):
    
    if(noise_gaussian_constant==0):
        folder = "./"+circuit_name +"/"+ str(qubits) +"_"+ str(n_params)+"_gd_gaussian_"+str(noise_gaussian_rate)
    else:
        folder = "./"+circuit_name +"/"+ str(qubits) +"_"+ str(n_params)+"_sgd_gaussian"
    if not os.path.exists(folder):
        os.makedirs(folder)
    theta = np.random.normal(0, gamma, n_params*n_time).reshape((n_time, n_params))
    for time in range(n_time):
        np.save(folder+"/params_init_"+str(time)+".npy", theta[time])
        loss, gradnorm, params = gd_optimzier(ansatz, circuit_name, H, excitations, qubits, theta[time], noise_gaussian_constant, noise_gaussian_rate, lr, iteration, n_check)
        np.save(folder+"/loss_"+str(time)+".npy", loss)
        np.save(folder+"/gradnorm_"+str(time)+".npy",  gradnorm)
        np.save(folder+"/params_final_"+str(time)+".npy", params)
    print("the training with gd gaussian ends")
    return 0


def chem_gd_zero(ansatz, circuit_name, H, excitations, qubits, n_params, gamma, noise_gaussian_constant, noise_gaussian_rate, lr, iteration, n_time, n_check):
    if(noise_gaussian_constant==0):
        folder = "./"+circuit_name +"/"+ str(qubits) +"_"+ str(n_params)+"_gd_zero_"+str(noise_gaussian_rate)
    else:
        folder = "./"+circuit_name +"/"+ str(qubits) +"_"+ str(n_params)+"_sgd_zero"
    if not os.path.exists(folder):
        os.makedirs(folder)
    theta = np.zeros(n_params*n_time).reshape((n_time, n_params))
    for time in range(n_time):
        np.save(folder+"/params_init_"+str(time)+".npy", theta[time])
        loss, gradnorm, params = gd_optimzier(ansatz, circuit_name, H, excitations, qubits, theta[time], noise_gaussian_constant, noise_gaussian_rate, lr, iteration, n_check)
        np.save(folder+"/loss_"+str(time)+".npy", loss)
        np.save(folder+"/gradnorm_"+str(time)+".npy",  gradnorm)
        np.save(folder+"/params_final_"+str(time)+".npy", params)
    print("the training with gd zero ends")            
    return 0


def chem_gd_uniform(ansatz, circuit_name, H, excitations, qubits, n_params, gamma, noise_gaussian_constant, noise_gaussian_rate, lr, iteration, n_time, n_check):
    if(noise_gaussian_constant==0):
        folder = "./"+circuit_name +"/"+ str(qubits) +"_"+ str(n_params)+"_gd_uniform_"+str(noise_gaussian_rate)
    else:
        folder = "./"+circuit_name +"/"+ str(qubits) +"_"+ str(n_params)+"_sgd_uniform"
    if not os.path.exists(folder):
        os.makedirs(folder)
    theta = np.random.uniform(0,2*pi, n_params*n_time).reshape((n_time, n_params))
    for time in range(n_time):
        np.save(folder+"/params_init_"+str(time)+".npy", theta[time])
        loss, gradnorm, params = gd_optimzier(ansatz, circuit_name, H, excitations, qubits, theta[time], noise_gaussian_constant, noise_gaussian_rate, lr, iteration, n_check)
        np.save(folder+"/loss_"+str(time)+".npy", loss)
        np.save(folder+"/gradnorm_"+str(time)+".npy",  gradnorm)
        np.save(folder+"/params_final_"+str(time)+".npy", params)
    print("the training with gd uniform ends")
    return 0


#--------------------------

def adam_optimzier(ansatz, circuit_name, H, excitations, qubits, theta, noise_gaussian_constant, noise_gaussian_rate, lr, iteration, n_check):
    beta_1 = 0.9
    beta_2 = 0.99
    epsilon = 0.00000001
    grad_func = qml.grad(ansatz)
    params = np.copy(theta)
    n_params = len(params)

    gradnorm, loss = np.zeros(iteration), np.zeros(iteration)
    m_params, v_params = np.zeros(n_params), np.zeros(n_params)

    t1 = datetime.now()
    noise_gamma = np.zeros(n_params)
    for j in range(n_params):
        noise_gamma[j] = noise_gaussian_constant + math.sqrt(1/(96*64*n_params))

    for j in range(iteration):
        loss[j] = ansatz(params, obs=H, qubits=qubits, excitations=excitations)
        params.requires_grad = True
        gradient_params = grad_func(params, obs=H, qubits=qubits, excitations=excitations)
        gradnorm[j] = la.norm(gradient_params)
        noise = [np.random.normal(0,noise_gamma[i],1)[0] for i in range(len(params))]
        if(noise_gaussian_rate==2):
            noise_gamma = [0.5*math.sqrt(gradient_params[i]**2) for i in range(len(params))]
        else:
            noise_gamma = [noise_gaussian_constant+noise_gaussian_rate*math.sqrt(gradient_params[i]**2/(96*64*len(params)*loss[j]**2)) for i in range(len(params))]
        gradient_params = gradient_params + noise
        m_params = beta_1*m_params + (1-beta_1)*gradient_params
        v_params = beta_2*v_params + (1-beta_2)*np.power(gradient_params, 2)
        m_params_hat = m_params / (1-np.power(beta_1, j+1))
        v_params_hat = v_params / (1-np.power(beta_2, j+1))
        params = params - lr*m_params_hat/(np.sqrt(v_params_hat)+epsilon)

        if(j%n_check==0):
            t2 = datetime.now()
            print(j, "th", " loss : %.6f" % loss[j], "gradnorm_theta: %.6f" % gradnorm[j], ",", (t2-t1).seconds, "seconds")
            t1 = t2
    return loss, gradnorm, params

def chem_adam_gaussian(ansatz, circuit_name, H, excitations, qubits, n_params, gamma, noise_gaussian_constant, noise_gaussian_rate, lr, iteration, n_time, n_check):
    
    if(noise_gaussian_constant==0):
        folder = "./"+circuit_name +"/"+ str(qubits) +"_"+ str(n_params)+"_adam_gaussian_"+str(noise_gaussian_rate)
    else:
        folder = "./"+circuit_name +"/"+ str(qubits) +"_"+ str(n_params)+"_sadam_gaussian"
    if not os.path.exists(folder):
        os.makedirs(folder)
    theta = np.random.normal(0, gamma, n_params*n_time).reshape((n_time, n_params))
    for time in range(n_time):
        np.save(folder+"/params_init_"+str(time)+".npy", theta[time])
        loss, gradnorm, params = adam_optimzier(ansatz, circuit_name, H, excitations, qubits, theta[time], noise_gaussian_constant, noise_gaussian_rate, lr, iteration, n_check)
        np.save(folder+"/loss_"+str(time)+".npy", loss)
        np.save(folder+"/gradnorm_"+str(time)+".npy",  gradnorm)
        np.save(folder+"/params_final_"+str(time)+".npy", params)
        print(time, "the training with adam gaussian ends")
    return 0


def chem_adam_zero(ansatz, circuit_name, H, excitations, qubits, n_params, gamma, noise_gaussian_constant, noise_gaussian_rate, lr, iteration, n_time, n_check):
    if(noise_gaussian_constant==0):
        folder = "./"+circuit_name +"/"+ str(qubits) +"_"+ str(n_params)+"_adam_zero_"+str(noise_gaussian_rate)
    else:
        folder = "./"+circuit_name +"/"+ str(qubits) +"_"+ str(n_params)+"_sadam_zero"
    if not os.path.exists(folder):
        os.makedirs(folder)
    theta = np.zeros(n_params*n_time).reshape((n_time, n_params))
    for time in range(n_time):
        np.save(folder+"/params_init_"+str(time)+".npy", theta[time])
        loss, gradnorm, params = adam_optimzier(ansatz, circuit_name, H, excitations, qubits, theta[time], noise_gaussian_constant, noise_gaussian_rate, lr, iteration, n_check)
        np.save(folder+"/loss_"+str(time)+".npy", loss)
        np.save(folder+"/gradnorm_"+str(time)+".npy",  gradnorm)
        np.save(folder+"/params_final_"+str(time)+".npy", params)
        print(time, "the training with adam zero ends")            
    return 0


def chem_adam_uniform(ansatz, circuit_name, H, excitations, qubits, n_params, gamma, noise_gaussian_constant, noise_gaussian_rate, lr, iteration, n_time, n_check):
    if(noise_gaussian_constant==0):
        folder = "./"+circuit_name +"/"+ str(qubits) +"_"+ str(n_params)+"_adam_uniform_"+str(noise_gaussian_rate)
    else:
        folder = "./"+circuit_name +"/"+ str(qubits) +"_"+ str(n_params)+"_sadam_uniform"
    if not os.path.exists(folder):
        os.makedirs(folder)
    theta = np.random.uniform(0,2*pi, n_params*n_time).reshape((n_time, n_params))
    for time in range(n_time):
        np.save(folder+"/params_init_"+str(time)+".npy", theta[time])
        loss, gradnorm, params = adam_optimzier(ansatz, circuit_name, H, excitations, qubits, theta[time], noise_gaussian_constant, noise_gaussian_rate, lr, iteration, n_check)
        np.save(folder+"/loss_"+str(time)+".npy", loss)
        np.save(folder+"/gradnorm_"+str(time)+".npy",  gradnorm)
        np.save(folder+"/params_final_"+str(time)+".npy", params)
        print(time, "the training with adam uniform ends")
    return 0
