import os
import random
import json
import numpy as np
import torch

def create_dirs(path):
    """create directories if path doesn't exist

    Arguments:
        path {str} -- path of the directory
    """
    if not os.path.exists(path):
        os.makedirs(path)


def set_seed(seed):
    np.random.seed(seed)
    random.seed(seed)
    torch.manual_seed(seed)

    # if you are suing GPU
    torch.cuda.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)


def mask_l2_loss(a, b, mask):
    return l2_loss(a[mask], b[mask])


def l2_loss(a, b):
    return ((a - b)**2).sum() / (len(a) * 2)


def get_device():
    if torch.cuda.is_available():
        device = torch.device("cuda:0")
        cuda_device = 0
    else:
        device = torch.device('cpu')
        cuda_device = -1
    return device, cuda_device


def convert_index_to_int(adj_lists):
    """Function to convert the node indices to int
    """
    new_adj_lists = {}
    for node, neigh in adj_lists.items():
        new_adj_lists[int(node)] = set(neigh)
    
    return new_adj_lists


def save_model(model, save_path):
    """The function is used to save the model

    Arguments:
        model {nn.Model} -- the model
        save_path {str} -- model save path
    """
    torch.save(model.state_dict(), save_path)


def setup_graph(graph_path, collapse=False):
    # load the adj list
    if collapse:
        pass
    else:
        adj_lists_path = os.path.join(graph_path, 'rw_adj_rel_lists.json')

    adj_lists = json.load(open(adj_lists_path))
    # fix this 
    adj_lists = convert_index_to_int(adj_lists)

    # load conceptnet embs
    concept_path = os.path.join(graph_path, 'concepts.pt')
    concept_embs = torch.load(concept_path)

    return adj_lists, concept_embs
