#!/usr/bin/env python
# -*- coding: utf-8 -*-

import torch
from torch import FloatTensor, LongTensor
from torch.nn import functional as F
from typing import Union

from . import AbstractWatermarkCode, AbstractReweight, AbstractScore


class Beta_WatermarkCode(AbstractWatermarkCode):
    def __init__(self, shuffle: LongTensor):
        self.shuffle = shuffle
        self.unshuffle = torch.argsort(shuffle, dim=-1)

    @classmethod
    def from_random(
        cls,
        rng: Union[torch.Generator, list[torch.Generator]],
        vocab_size: int,
    ):
        if isinstance(rng, list):
            batch_size = len(rng)
            shuffle = torch.stack(
                [
                    torch.randperm(vocab_size, generator=rng[i], device=rng[i].device)
                    for i in range(batch_size)
                ]
            )
        else:
            shuffle = torch.randperm(vocab_size, generator=rng, device=rng.device)
        return cls(shuffle)


class Beta_Reweight(AbstractReweight):
    watermark_code_type = Beta_WatermarkCode

    def __init__(self, beta: float = 0.1):
        self.beta = beta

    def __repr__(self):
        return f"Beta_Reweight(beta={self.beta})"

    def reweight_logits(
        self, code: AbstractWatermarkCode, p_logits: FloatTensor
    ) -> FloatTensor:
        # s_ means shuffled
        s_p_logits = torch.gather(p_logits, -1, code.shuffle)
        s_log_cumsum = torch.logcumsumexp(s_p_logits, dim=-1)
        # normalize the log_cumsum to force the last element to be 0
        s_log_cumsum = s_log_cumsum - s_log_cumsum[..., -1:]
        s_cumsum = torch.exp(s_log_cumsum)
        s_p = F.softmax(s_p_logits, dim=-1)

        #  _t = torch.cat([torch.zeros_like(s_cumsum[..., :1]), s_cumsum], dim=-1)
        boundary = torch.argmax((s_cumsum > 1 / 2).to(torch.int), dim=-1, keepdim=True)
        p_boundary = torch.gather(s_p, -1, boundary)
        portion_in_right = (torch.gather(s_cumsum, -1, boundary) - 1 / 2) / p_boundary
        portion_in_right = torch.clamp(portion_in_right, 0, 1)

        s_all_portion_in_right = (s_cumsum > 1 / 2).type_as(p_logits)
        s_all_portion_in_right.scatter_(-1, boundary, portion_in_right)
        s_shift_logits = torch.log(
            (self.beta * 2) * (1 - s_all_portion_in_right)
            + (1 - self.beta) * 2 * s_all_portion_in_right
        )
        shift_logits = torch.gather(s_shift_logits, -1, code.unshuffle)
        return p_logits + shift_logits
        #
        #  hi = cumsum
        #  lo = torch.cat([torch.zeros_like(cumsum[..., :1]), cumsum[..., :-1]], dim=-1)
        #
        #  s_p_logits = torch.gather(p_logits, -1, code.shuffle)
        #  cumsum = torch.cumsum(F.softmax(s_p_logits, dim=-1), dim=-1)
        #  portion_in_left = cumsum < 1 / 2
        #  reweighted_cumsum = torch.where(
        #      cumsum < 1 / 2,
        #      (1 - self.gamma) * cumsum,
        #      -self.gamma + (1 + self.gamma) * cumsum,
        #  )
        #  suffled_rewighted_p = torch.diff(
        #      reweighted_cumsum,
        #      dim=-1,
        #      prepend=torch.zeros_like(reweighted_cumsum[..., :1]),
        #  )
        #  rewighted_p = torch.gather(suffled_rewighted_p, -1, code.unshuffle)
        #  reweighted_p_logits = torch.log(rewighted_p)
        #  return reweighted_p_logits
