import numpy as np
from gym import utils
from gym.envs.mujoco import mujoco_env
import cv2
import os


class _FrameBufferEnv:
    def __init__(self, past_frames):
        self._initialized = False
        self._past_frames = past_frames

    def _init_buffer(self, im):
        self._im_size = im.shape
        self._reset_buffer()

    def _reset_buffer(self, ):
        # self._frames_buffer = np.zeros([self._past_frames] + list(self._im_size))
        self._frames_buffer = np.zeros([self._past_frames] + list(self._im_size)).astype('uint8')

    def _update_buffer(self, im):
        # self._frames_buffer = np.concatenate([np.expand_dims(im, 0), self._frames_buffer[:-1, :, :, :]], axis=0)
        self._frames_buffer = np.concatenate([np.expand_dims(im.astype('uint8'), 0),
                                              self._frames_buffer[:-1, :, :, :]], axis=0).astype('uint8')


class _CustomInvertedPendulumEnv(mujoco_env.MujocoEnv, utils.EzPickle, _FrameBufferEnv):
    def __init__(self, size=(32, 32), color_permutation=[0, 1, 2],
                 smoothing_factor=0.0, past_frames=4, not_done=True):
        self._size = size
        self._not_done = not_done
        self._color_permutation = color_permutation
        self._smooth = 1.0 - smoothing_factor
        _FrameBufferEnv.__init__(self, past_frames)
        utils.EzPickle.__init__(self)
        path_to_xml = os.path.join(os.path.dirname(__file__), 'assets/custom_inverted_pendulum.xml')
        mujoco_env.MujocoEnv.__init__(self, path_to_xml, 2)

    def step(self, a):
        reward = 1.0
        self.do_simulation(a, self.frame_skip)
        ob = self._get_obs()
        notdone = np.isfinite(ob).all() and (np.abs(ob[1]) <= .2)
        done = not notdone
        if done and self._not_done:
            done = False
            reward = 0.0
        return ob, reward, done, {}

    def reset_model(self):
        qpos = self.init_qpos + self.np_random.uniform(size=self.model.nq, low=-0.01, high=0.01)
        qvel = self.init_qvel + self.np_random.uniform(size=self.model.nv, low=-0.01, high=0.01)
        self.set_state(qpos, qvel)
        if self._initialized:
            self._reset_buffer()
        return self._get_obs()

    def _get_obs(self):
        return np.concatenate([self.sim.data.qpos, self.sim.data.qvel]).ravel()

    def viewer_setup(self):
        v = self.viewer
        v.cam.trackbodyid = 0
        v.cam.distance = self.model.stat.extent

    def get_ims(self):
        raw_im = (self.render(mode='rgb_array'))[:, :, self._color_permutation] * self._smooth
        im = cv2.resize(raw_im, dsize=self._size, interpolation=cv2.INTER_AREA)
        if not self._initialized:
            self._init_buffer(im)
            self._initialized = True
        self._update_buffer(im)
        curr_frames = self._frames_buffer.astype('uint8')
        return curr_frames


class AgentInvertedPendulumEnv(_CustomInvertedPendulumEnv):
    def __init__(self, ):
        super(AgentInvertedPendulumEnv, self).__init__(size=(32, 32),
                                                       color_permutation=[0, 1, 2],
                                                       smoothing_factor=0.0,
                                                       past_frames=4,
                                                       not_done=True)


class ExpertInvertedPendulumEnv(_CustomInvertedPendulumEnv):
    def __init__(self, ):
        super(ExpertInvertedPendulumEnv, self).__init__(size=(32, 32),
                                                        color_permutation=[2, 1, 0],
                                                        smoothing_factor=0.1,
                                                        past_frames=4,
                                                        not_done=True)


class _CustomInvertedDoublePendulumEnv(mujoco_env.MujocoEnv, utils.EzPickle, _FrameBufferEnv):  # TODO
    def __init__(self, size=(32, 32), color_permutation=[0, 1, 2],
                 smoothing_factor=0.0, past_frames=4, not_done=True):
        self._size = size
        self._not_done = not_done
        self._failure = False
        self._color_permutation = color_permutation
        self._smooth = 1.0 - smoothing_factor
        _FrameBufferEnv.__init__(self, past_frames)
        utils.EzPickle.__init__(self)
        path_to_xml = os.path.join(os.path.dirname(__file__), 'assets/custom_inverted_double_pendulum.xml')
        mujoco_env.MujocoEnv.__init__(self, path_to_xml, 2)

    def step(self, action):
        self.do_simulation(action, self.frame_skip)
        ob = self._get_obs()
        x, _, y = self.sim.data.site_xpos[0]
        dist_penalty = 0.01 * x ** 2 + (y - 2) ** 2
        v1, v2 = self.sim.data.qvel[1:3]
        vel_penalty = 1e-3 * v1 ** 2 + 5e-3 * v2 ** 2
        alive_bonus = 10
        r = alive_bonus - dist_penalty - vel_penalty
        done = bool(y <= 1)
        if done and self._not_done:
            done = False
            self._failure = True
        if self._failure:
            r = 0.0
        return ob, r, done, {}

    def reset_model(self):
        self._failure = False
        self.set_state(
            self.init_qpos + self.np_random.uniform(low=-.1, high=.1, size=self.model.nq),
            self.init_qvel + self.np_random.randn(self.model.nv) * .1
        )
        if self._initialized:
            self._reset_buffer()
        return self._get_obs()

    def _get_obs(self):
        return np.concatenate([
            self.sim.data.qpos[:1],
            np.sin(self.sim.data.qpos[1:]),
            np.cos(self.sim.data.qpos[1:]),
            np.clip(self.sim.data.qvel, -10, 10),
            np.clip(self.sim.data.qfrc_constraint, -10, 10)
        ]).ravel()

    def get_ims(self):
        raw_im = (self.render(mode='rgb_array'))[:, :, self._color_permutation] * self._smooth
        im = cv2.resize(raw_im, dsize=self._size, interpolation=cv2.INTER_AREA)
        if not self._initialized:
            self._init_buffer(im)
            self._initialized = True
        self._update_buffer(im)
        curr_frames = self._frames_buffer.astype('uint8')
        return curr_frames

class AgentInvertedDoublePendulumEnv(_CustomInvertedDoublePendulumEnv):
    def __init__(self, ):
        super(AgentInvertedDoublePendulumEnv, self).__init__(size=(32, 32),
                                                             color_permutation=[0, 1, 2],
                                                             smoothing_factor=0.0,
                                                             past_frames=4,
                                                             not_done=True)


class ExpertInvertedDoublePendulumEnv(_CustomInvertedDoublePendulumEnv):
    def __init__(self, ):
        super(ExpertInvertedDoublePendulumEnv, self).__init__(size=(32, 32),
                                                              color_permutation=[2, 1, 0],
                                                              smoothing_factor=0.1,
                                                              past_frames=4,
                                                              not_done=True)


