#!/usr/bin/env python3
import json
import argparse

def fix_duplicate_idx(input_file, output_file):
    """
    修复JSON文件中重复的idx问题，只保留第一次出现的idx不变，其他重复的进行重新编码
    
    参数:
    input_file (str): 输入JSON文件路径
    output_file (str): 输出JSON文件路径
    """
    # 读取JSON文件
    with open(input_file, 'r', encoding='utf-8') as f:
        data = json.load(f)
    
    # 如果数据不是列表格式，则直接返回
    if not isinstance(data, list):
        print("错误：输入的JSON不是列表格式")
        return False
    
    # 用于跟踪已经出现过的idx
    seen_idx = set()
    
    # 用于记录下一个可用的idx
    next_available_idx = 0
    
    # 遍历所有数据项
    for item in data:
        # 确保数据项包含idx字段
        if 'idx' not in item:
            continue
        
        current_idx = item['idx']
        
        # 找到最大的idx值用于确定下一个可用的idx
        if isinstance(current_idx, int) and current_idx >= next_available_idx:
            next_available_idx = current_idx + 1
        
        # 如果idx已经存在，则分配一个新的idx
        if current_idx in seen_idx:
            item['idx'] = next_available_idx
            next_available_idx += 1
        else:
            seen_idx.add(current_idx)
    
    # 保存修改后的数据到新文件
    with open(output_file, 'w', encoding='utf-8') as f:
        json.dump(data, f, ensure_ascii=False, indent=2)
    
    print(f"处理完成，结果已保存到：{output_file}")
    return True

def main():
    parser = argparse.ArgumentParser(description='修复JSON文件中重复的idx问题')
    parser.add_argument('input_file', type=str, help='输入JSON文件路径')
    parser.add_argument('output_file', type=str, help='输出JSON文件路径')
    
    args = parser.parse_args()
    
    fix_duplicate_idx(args.input_file, args.output_file)

if __name__ == "__main__":
    main()