import argparse
import json
import logging
import pprint

import pandas as pd

from pathlib import Path


def define_logger(debug: bool = False) -> None:
    """Define a Logger.

    Args:
        debug (bool, optional): Specifies the debugging mode. Defaults to False.
    """
    log_format = "[%(asctime)s] [%(levelname)s] %(message)s"
    level = logging.DEBUG if debug else logging.INFO

    ## Save log.
    logging.basicConfig(level=level, format=log_format)


def print_config_rank_0(config: argparse.Namespace, rank: int = 0) -> None:
    """Display configuration items if rank is 0.

    Args:
        config (argparse.Namespace): The class that contains the configuration item
        rank (int, optional): Rank of the calling process. Defaults to 0.
    """
    if rank <= 0:
        pprint.PrettyPrinter(indent=4, sort_dicts=False).pprint(vars(config))


def print_rank_0(msg: str, rank: int = 0) -> None:
    """Print message if rank is 0.

    Args:
        msg (str): A message to print
        rank (int, optional): Rank of the calling process. Defaults to 0.
    """
    if rank <= 0:
        print(msg)


def save_results(out: pd.DataFrame, save_path: str) -> None:
    """Save the experiment results as a json file.

    Args:
        out (pd.DataFrame]): Experiment result
        save_path (str): The path to save
    """
    ## Save.
    Path(save_path).parent.mkdir(parents=True, exist_ok=True)

    out = json.loads(out.to_json(orient="records"))
    with open(save_path, "w", encoding="utf-8") as f:
        json.dump(out, f, indent=4)


def load_results(save_path: str) -> pd.DataFrame:
    """Load the texts generated by extract.py.

    Args:
        save_path (str): Path where results are stored

    Returns:
        pd.DataFrame: Loaded saved results
    """
    ## Results.
    out = pd.read_json(save_path, orient="records")
    return out
