"""
author: Anonymous

Large Scale Problem Generator using a Divide and Conquer Method
"""

import os
import sys
import inspect
import json

currentdir = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
parentdir = os.path.dirname(currentdir)
sys.path.insert(0, parentdir) 

import numpy as np
from dataclasses import dataclass
import tyro

from scheduling.environment import SchedulingEnvironment
from utils.utils import set_seed
    

@dataclass
class SchedulingProblemGeneratorArgs:
    shared_folder: str = "data/problem_set_r40_t80_s0_f10_w25_euc_2000_uni"
    """the folder where the shared paths are stored"""
    seed: int = 10
    """the seed for the random number generator"""
    config_file: str = "data/problem_r10_t20_s0_f10_w25_config.json"
    """the configuration file for the problem"""
    start_problem: int = 1
    """the starting problem number"""
    end_problem: int = 200
    """the ending problem number"""
    file_prefix: str = "problem_"
    """the prefix for the problem file names"""
    travel_mode: str = "euclidean"
    """the travel mode for the environment (euclidean, manhattan or rrg). 
    RRG Mode handles shared paths and includes additional constraint 
    checking to allow for the presence of paths from each robot and task 
    location to the task locations."""
    number_of_agents: int = 40
    """the number of agents in the environment"""
    number_of_tasks: int = 80
    """the number of tasks in the environment"""
    tmp_location: str = "data/tmp_data_generator_large_scale"
    """the temporary location for the subset problems"""

if __name__ == "__main__":
    print("Generating problems for the scheduling environment.")
    args = tyro.cli(SchedulingProblemGeneratorArgs)
    set_seed(args.seed)
    # if the folder does not exist, create it
    if not os.path.exists(args.shared_folder):
        os.makedirs(args.shared_folder)
        
    if not os.path.exists(args.tmp_location):
        os.makedirs(args.tmp_location)
    with open(args.config_file, 'r') as f:
        config = json.load(f)
        subset_num_agents = config['num_agents'][0]
        subset_num_tasks = config['num_tasks'][0]
    # check if the number of agents and tasks are divisible by the subset number of agents and tasks
    assert(args.number_of_agents % subset_num_agents == 0)
    assert(args.number_of_tasks % subset_num_tasks == 0)
    assert(args.number_of_agents // subset_num_agents == args.number_of_tasks // subset_num_tasks)
    n = args.number_of_agents // subset_num_agents
    for i in range(args.start_problem, args.end_problem + 1):
        seed = args.seed + i
        set_seed(seed) # because the model is likely to fail due to size of the larger scale model, the seed is modified to prevent the problems from being the same between different instances.
        ids = []
        obstacles = None
        if os.path.exists(os.path.join(args.shared_folder, f"{args.file_prefix}{str(i).zfill(5)}.json")):
            print(f"Problem {i} already exists. Skipping...")
            continue
        for j in range(n):
            print(f"Generating problem {i} subset {j}...")
            id = (i - 1) * n + j
            save_path = os.path.join(args.tmp_location, f"{args.file_prefix}{str(id).zfill(5)}.json")
            env = SchedulingEnvironment(save_path, generator_config=args.config_file, mode=args.travel_mode, verbose=True, obstacles=obstacles)
            env.save()
            print(f"Problem {i} subset {j} is generated successfully.")
            ids.append(id)
            obstacles = env.obstacles # retain the same obstacle set for the next subset problem for synched path planning and task completion
        # combine the subset problems into a single problem (divide and conquer)
        save_paths = [os.path.join(args.tmp_location, f"{args.file_prefix}{str(id).zfill(5)}.json") for id in ids]
        env = SchedulingEnvironment.combine_problems(save_paths, os.path.join(args.shared_folder, f"{args.file_prefix}{str(i).zfill(5)}.json"), args.travel_mode)
        print(f"Problem {i} is generated successfully.")
        
    print("All problems are generated successfully.")
