"""
author:  Anonymous, Anonymous
"""
import os
import sys
import inspect

currentdir = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
parentdir = os.path.dirname(currentdir)
sys.path.insert(0, parentdir) 

import numpy as np
import torch

import time
from gurobipy import Model, Env, GRB, LinExpr

class MILP_Solver():
    def __init__(self):
        self.name = "MILP_Solver"
        
    def set_environment(self, env, rerun=False):
        # read the environment location and get the schedule file
        self.env = env
        schedule = env._get_schedule_from_file()
        if schedule is None or rerun:
            feasible, status, reward, self.schedule, _ = solve_with_MILP(env, time_limit=60*60)
            if not feasible:
                raise ValueError("The problem is infeasible")
        else:
            self.schedule = schedule
            
    def get_action(self, observation, greedy=False):
        num_tasks = len(self.schedule)
        unassigned_tasks = observation['task_to_task_select'].source_nodes
        num_tasks_assigned = num_tasks - len(unassigned_tasks)
        return self.schedule[num_tasks_assigned], None, None, None
    
    def get_agent_probs(self, observation, greedy=True, adaptive_temperature=False):
        """Get agent probabilities
        Args:
            x (dict): Input data Observation
            greedy (bool): Greedy action
            adaptive_temperature (bool): Adaptive temperature
        Returns:
            tuple: Tuple of agent ID, agent probabilities, agent log probabilities
        """
        num_tasks = len(self.schedule)
        unassigned_tasks = observation['task_to_task_select'].source_nodes
        num_tasks_assigned = num_tasks - len(unassigned_tasks)
        _, agent = self.schedule[num_tasks_assigned]
        
        agent_probs = np.zeros(self.env.num_agents)
        agent_probs[agent] = 1.0
        agent_log_probs = np.log(agent_probs + 1e-8)
        return agent, agent_probs, agent_log_probs
    
    def get_task_probs(self, observation, agent_id, greedy=True, adaptive_temperature=False):
        """Get task probabilities
        Args:
            x (dict): Input data Observation
            agent_id (int): Agent ID
        Returns:
            tuple: Tuple of task ID, task probabilities, task log probabilities
        """
        num_tasks = len(self.schedule)
        unassigned_tasks = observation['task_to_task_select'].source_nodes
        num_tasks_assigned = num_tasks - len(unassigned_tasks)
        task, _ = self.schedule[num_tasks_assigned]
        
        task_probs = np.zeros(len(unassigned_tasks))
        task_probs[task] = 1.0
        task_log_probs = np.log(task_probs + 1e-8)
        return task, task_probs, task_log_probs

class MILP_Partial_Solver():
    def __init__(self):
        """ MILP Solver with partial schedule """
        self.name = "MILP_Partial_Solver"
        self.schedule = []
        self.feasible = True
        self.buffer = None
    
    def set_environment(self, env, rerun):
        self.step = 0
        self.env = env
        self.feasible = True
        self.partial_schedule = []
        self.schedule = []
        schedule = env._get_schedule_from_file()
        if schedule is None:
            feasible, status, reward, self.schedule, _ = solve_with_MILP(env, time_limit=60*60)
            if not feasible:
                raise ValueError("The problem is infeasible")
        else:
            self.schedule = schedule
            
    def get_action(self, observation, greedy=False):
        """ Get action based on the partial schedule """
        step = self.schedule[self.step]
        self.step += 1
        return step, None, None, None

    def get_action_from_partial(self, observation, greedy=False):
        """ Get action based on the partial schedule """
        if self.buffer is None:
            feasible, _, _, schedule, _ = solve_with_MILP_given_partial_schedule(self.env, self.partial_schedule)
            self.buffer = schedule[self.step]
        step = self.buffer
        self.buffer = None # TODO: Speed this up a bit.
        return step, None, None, None
            
        # if not self.feasible:
        #     # MILP will fail
        #     pass
        # else:
        #     self.feasible, _, _, schedule, _ = solve_with_MILP_given_partial_schedule(self.env, self.partial_schedule)
        #     if not self.feasible:
        #         # random
        #         pass
        #     self.partial_schedule.append(schedule[self.step])
        #     self.step += 1
        #     return self.partial_schedule[-1], None, None, None
        
    def get_agent_probs(self, observation, greedy=True, adaptive_temperature=False):
        """ Get agent probabilities based on the partial schedule """
        step = len(self.partial_schedule)
        if self.buffer is None:
            self.feasible, _, _, schedule, _ = solve_with_MILP_given_partial_schedule(self.env, self.partial_schedule)
            self.buffer = schedule[step]
        _, agent = self.buffer
        agent_probs = np.zeros(self.env.num_agents)
        agent_probs[agent] = 1.0
        agent_log_probs = np.log(agent_probs + 1e-8)
        return agent, torch.tensor([agent_probs]), agent_log_probs
        
    def get_task_probs(self, observation, agent_id, greedy=True, adaptive_temperature=False):
        """ Get task probabilities based on the partial schedule """
        step = len(self.partial_schedule)
        if self.buffer is None:
            self.feasible, _, _, schedule, _ = solve_with_MILP_given_partial_schedule(self.env, self.partial_schedule)
            self.buffer = schedule[step]
        task, agent = self.buffer
        self.partial_schedule.append((task, agent))
        self.buffer = None # reset buffer
        
        unassigned_tasks = observation['task_to_task_select'].source_nodes
        # get the index of the task in the unassigned tasks
        task_assignment_id = np.where(unassigned_tasks == task)[0][0]
        # task_id = unassigned_tasks[task_assignment_id]
        
        task_probs = np.zeros(len(unassigned_tasks))
        task_probs[task_assignment_id] = 1.0
        log_probs = np.log(task_probs + 1e-8)
        return task, torch.tensor([task_probs]), log_probs
        
        
        

def get_constraints(env, verbose=False):
    M = 5
    M = env.deadline * M
    
    e = Env()
    e.setParam("OutputFlag", 0)

    m = Model("Deterministic-Solver", env=e)
    
    num_units = env.num_agents
    num_tasks = env.num_tasks
    
    # Aaj = assignment variable (indicates if Task j is assigned to agent a)
    for a in range(1, num_units+1):
        for j in range(num_tasks + 1):
            Aaj = "A%03d%03d" % (a, j)
            m.addVar(vtype=GRB.BINARY, name=Aaj)

    # Xjk = sequencing variable (indicates if task j is assigned to agent a before task k)
    for j in range(0, num_tasks + 1):
        for k in range(num_tasks + 1):
            Xjk = "X%03d%03d" % (j, k)
            m.addVar(vtype=GRB.BINARY, name=Xjk)

    # Adding temporal variables -

    # sk = start time of task k
    for k in range(num_tasks+1):
        sk = "s%03d" % (k)
        m.addVar(vtype=GRB.INTEGER, name=sk)

    # fk = finish time of task k
    for k in range(num_tasks+1):
        fk = "f%03d" % (k)
        m.addVar(vtype=GRB.INTEGER, name=fk)

    m.addVar(vtype=GRB.INTEGER, name="mk")
    m.update()

    # Adding Constraints

    ## Adding sequencing constraints -
    for j in range(1, num_tasks + 1):
        for k in range(1, num_tasks + 1):
            Xjk = "X%03d%03d" % (j, k)
            Xjk_ = m.getVarByName(Xjk)
            Xkj = "X%03d%03d" % (k, j)
            Xkj_ = m.getVarByName(Xkj)
            sum_k = LinExpr()
            sum_j = 0
            for a in range(1, num_units+1):
                Aak = "A%03d%03d" % (a, k)
                Aak_ = m.getVarByName(Aak)
                Aaj = "A%03d%03d" % (a, j)
                Aaj_ = m.getVarByName(Aaj)
                sum_k.add(Aak_)
                sum_j += Aaj_
            #print(sum_j, sum_k)
            if j != k: # if j and k are not the same task, then they should be ordered if they are assigned to the same agent
                m.addConstr((sum_j*sum_k) == Xjk_ + Xkj_)
            if k !=0 : # if k is not the start node, then it should be assigned to a single agent
                m.addConstr(sum_k == 1)
                
    # Adding start and finish time constraints
    mk_ = m.getVarByName("mk")
    for k in range(1, num_tasks + 1):
        sk = "s%03d" % (k)
        sk_ = m.getVarByName(sk)
        fk = "f%03d" % (k)
        fk_ = m.getVarByName(fk)
        
        task = env.tasks[k-1]
        m.addConstr(sk_ >= task.start_time)
        m.addConstr(fk_ <= task.end_time)
        m.addConstr(fk_ <= mk_)
    
    for a in range(1, num_units+1):
        Aa0 = "A%03d000" % (a)
        Aa0_ = m.getVarByName(Aa0)
        m.addConstr(Aa0_ == 1)
        for j in range(0, num_tasks + 1):
            Xj0 = "X%03d000" % (j)
            Xj0_ = m.getVarByName(Xj0)
            m.addConstr(Xj0_ == 0)
            if j !=0:
                X0j = "X000%03d" % (j)
                X0j_ = m.getVarByName(X0j)
                m.addConstr(X0j_ == 1)
                
            task_j = j - 1
            if task_j in env.wait_time_constraints:
                wait_times = {dependency['prerequisite']: dependency['wait_time'] for dependency in env.wait_time_constraints[task_j]}
            else:
                wait_times = None
            for k in range(1, num_tasks + 1):
                if j == k:
                    continue
                Xjk = "X%03d%03d" % (j, k)
                Xjk_ = m.getVarByName(Xjk)
                Aak = "A%03d%03d" % (a, k)
                Aak_ = m.getVarByName(Aak)
                Aaj = "A%03d%03d" % (a, j)
                Aaj_ = m.getVarByName(Aaj)
                sk = "s%03d" % (k)
                sk_ = m.getVarByName(sk)
                fk = "f%03d" % (k)
                fk_ = m.getVarByName(fk)
                sj = "s%03d" % (j)
                sj_ = m.getVarByName(sj)
                fj = "f%03d" % (j)
                fj_ = m.getVarByName(fj)
                
                unit_id = a - 1
                task_k = k - 1
                if j == 0:
                    tt = env._get_agent_task_travel_time(unit_id, task_k)
                else:
                    tt = env._get_task_task_travel_time(unit_id, task_j, task_k)
                    
                dur = env._get_duration(unit_id, task_k)
                # print(__file__, "Dur:", dur, distance,)
                if wait_times is not None and task_k in wait_times:
                    # There is a wait time constraint between the two tasks, as such the constraint we are adding is sk_ > fj_
                    # m.addConstr(sj_ >= fk_) # j comes after k
                    # Replace with setting the booleans based on order
                    Xjk = "X%03d%03d" % (j, k)
                    Xjk_ = m.getVarByName(Xjk)
                    Xkj = "X%03d%03d" % (k, j)
                    Xkj_ = m.getVarByName(Xkj)
                    # Constraints to ensure j comes after k
                    # m.addConstr(Xjk_ == 0)
                    # m.addConstr(Xkj_ == 1)
                    # m.addConstr(sj_ >= fk_)
                    # # When we add the actual wait_time duration, we can add the following constraint
                    # wait_time = env.wait_targets[task_j][task_k]
                    wait_time = wait_times[task_k]
                    m.addConstr(sj_ - fk_ >= wait_time)
                # if the j and k are assigned to a, and j comes before k, then account for the travel time
                m.addConstr((sk_ - fj_) >= -M * (3 - Aak_ - Aaj_ - Xjk_) + tt)
                # if the task k is assigned to a, the task k should take more than dur amount of time
                m.addConstr((fk_ - sk_) >= -M * (1 - Aak_) + dur)
    
    ## Adding objective - 
    # if obj == 1:
    # Reward = LinExpr()
    # for a in range(1, num_units+1):
    #     for j in range(1, num_tasks + 1):
    #         Aaj = "A%03d%03d" % (a, j)
    #         Aaj_ = m.getVarByName(Aaj)
    #         fj = "f%03d" % (j)
    #         fj_ = m.getVarByName(fj)
            
    #         end_time = (self.stn.edges[(S0, fj)]['weight'] - self.eps)
    #         r_val = 1*(1 - (fj_/end_time))
    #         Reward += r_val*Aaj_
    # m.setObjective(Reward, GRB.MAXIMIZE)
    return m, mk_

def solve_with_MILP(env, verbose=False, time_limit=None, threads=None):
    """ Solve the MILP
    Args:
        env (Environment): Environment object
        verbose (bool): Verbose mode
        time_limit (int): Time limit for the MILP, in seconds
    Returns:
        bool: Feasibility of the MILP
        int: Status of the MILP
        float: Objective value of the MILP
        list: Schedule
        float: Duration of the MILP
    """
    num_units = env.num_agents
    num_tasks = env.num_tasks
    m, mk_ = get_constraints(env, verbose)
    
    if time_limit is not None:
        m.setParam("TimeLimit", time_limit)
    if threads is not None:
        m.setParam("Threads", threads)
        m.setParam("Presolve", 0)
        m.setParam("LazyConstraints", 0)
        seed = np.random.randint(0, 100000)
        m.setParam("Seed", int(seed))
        m.setParam("Presolve", 0)
        m.setParam("Cuts", 0)
        m.setParam("Cache", 0)
        m.addConstr(mk_ <= 0)
        m.remove(m.getConstrs()[-1])
    m.Params.PoolSolutions = 1
    # else: 
    m.setObjective(mk_, GRB.MINIMIZE)

    time_start = time.perf_counter()
    m.feasRelaxS(0, False, False, True)
    m.optimize()
    time_end = time.perf_counter()
    duration = time_end - time_start
    
    if verbose:
        # print("Status is ####", m.Status)
        status = m.Status
        if status == GRB.UNBOUNDED:
            print('The model cannot be solved because it is unbounded')
            # sys.exit(0)
        if status == GRB.OPTIMAL:
            print('The optimal objective is %g' % m.ObjVal)
            # sys.exit(0)
        if status != GRB.INF_OR_UNBD and status != GRB.INFEASIBLE:
            print('Optimization was stopped with status %d' % status)
              
            # sys.exit(0)
        if status == GRB.TIME_LIMIT:
            # the model has reached the time limit, get partially optimal solution
            print('The model has reached the time limit')
        if status == GRB.INFEASIBLE:
            # do IIS
            print('The model is infeasible; computing IIS')
            m.computeIIS()
            if m.IISMinimal:
                print('IIS is minimal\n')
            else:
                print('IIS is not minimal\n')
            print('\nThe following constraint(s) cannot be satisfied:')
            for c in m.getConstrs():
                if c.IISConstr:
                    print('%s' % c.ConstrName)
    # if the model is infeasible and time limit is not set, return False
    if m.Status == GRB.INFEASIBLE and time_limit is None:
        return False, m.Status, None, None, duration
    
    if m.SolCount == 0:
        return False, m.Status, None, None, duration
    
    # build the schedule using assignment map
    obj = m.getObjective()
    
    Xjk_mat = np.zeros((num_tasks + 1, num_tasks + 1))
    Aaj_mat = np.zeros((num_units, num_tasks + 1))
    fk_mat = np.zeros(num_tasks)
    for a in range(1, num_units+1):
        for j in range(0, num_tasks + 1):
            Aaj = "A%03d%03d" % (a, j)
            Aaj_ = m.getVarByName(Aaj)
            # if j is assigned to a, set the value in the matrix
            # if X exists as a parameter of Aaj_ and is equal to 1, set the value in the matrix
            # if Aaj_.X == 1: Aaj_mat[a-1][j] = Aaj_.X # this is wrong
            if hasattr(Aaj_, 'X') and Aaj_.X == 1: Aaj_mat[a-1][j] = Aaj_.X
            for k in range(1, num_tasks + 1):
                Xjk = "X%03d%03d" % (j, k)
                Xjk_ = m.getVarByName(Xjk)
                fk = "f%03d" % (k)
                fk_ = m.getVarByName(fk)
                # print(fk_)
                if hasattr(fk_, 'Xn'):
                    fk_mat[k-1] = fk_.Xn
                # if j comes after k, set the value in the matrix
                if hasattr(Xjk_, 'X') and Xjk_.X == 1: Xjk_mat[j][k] = Xjk_.X
    
    # get start time for each task
    task_start_times = np.zeros((num_tasks))
    for k in range(1, num_tasks + 1):
        sk = "s%03d" % (k)
        sk_ = m.getVarByName(sk)
        if hasattr(sk_, 'Xn'):
            task_start_times[k-1] = sk_.Xn
        
    # print('\n'.join([' '.join([str(int(Xjk_mat[i][j])) for j in range(num_tasks+1)]) for i in range(num_tasks+1)]))
    schedule = extract_task_assignment(Aaj_mat[0:, 1:], Xjk_mat[1:, 1:], task_start_times)

    if verbose:
        print("Schedule - ", schedule)
        print(obj.getValue())
        m.write("my_model.lp")
        print("\nXjk - \n", Xjk_mat)
        print("\n Aaj - \n", Aaj_mat)
        print("\n fk - \n", fk_mat)
    
    return True, m.Status, obj.getValue(), schedule, duration

def warmstart_MILP(env, schedule, verbose=False):
    """ Solve the MILP given a schedule to warmstart the optimizer
    Args:
        env (SchedulingEnvironment): Environment
        schedule (list): Schedule to warmstart the optimizer
    Returns:
        tuple: Tuple of feasible, status, reward, schedule, duration
    """
    num_units = env.num_agents
    num_tasks = env.num_tasks
    
    m, mk_ = get_constraints(env, verbose)
    
    # access the variables and add the warmstart constraints
    # get an assignment map
    assignment_map = {}
    for i, (task, agent) in enumerate(schedule):
        if agent not in assignment_map:
            assignment_map[agent] = []
        assignment_map[agent].append(task)
    
    # construct the task assignment
    for agent, tasks in assignment_map.items():
        for i, task in enumerate(tasks):
            Aaj = "A%03d%03d" % (agent+1, task+1)
            Aaj_ = m.getVarByName(Aaj)
            Aaj_.start = 1 # set assignment warmstart for agent task pair
            X0j_ = m.getVarByName("X000%03d" % (task+1))
            X0j_.start = 1 # set the 0th task warmstart
            for j, task in enumerate(tasks[0:i]):
                Xjk = "X%03d%03d" % (tasks[i-1]+1, task+1)
                Xjk_ = m.getVarByName(Xjk)
                # j comes before k
                Xjk_.start = 1
    # time to solve the MILP
    time_start = time.perf_counter()
    # solve the MILP
    m.optimize()
    time_end = time.perf_counter()
    
    duration = time_end - time_start
    
    if verbose:
        # print("Status is ####", m.Status)
        status = m.Status
        if status == GRB.UNBOUNDED:
            print('The model cannot be solved because it is unbounded')
            # sys.exit(0)
        if status == GRB.OPTIMAL:
            print('The optimal objective is %g' % m.ObjVal)
            # sys.exit(0)
        if status != GRB.INF_OR_UNBD and status != GRB.INFEASIBLE:
            print('Optimization was stopped with status %d' % status)
            # sys.exit(0)
        if status == GRB.INFEASIBLE:
            # do IIS
            print('The model is infeasible; computing IIS')
            m.computeIIS()
            if m.IISMinimal:
                print('IIS is minimal\n')
            else:
                print('IIS is not minimal\n')
            print('\nThe following constraint(s) cannot be satisfied:')
            for c in m.getConstrs():
                if c.IISConstr:
                    print('%s' % c.ConstrName)
    if m.Status == GRB.INFEASIBLE:
        return False, m.Status, None, None, duration
    
    # build the schedule using assignment map
    obj = m.getObjective()
    Xjk_mat = np.zeros((num_tasks + 1, num_tasks + 1))
    Aaj_mat = np.zeros((num_units, num_tasks + 1))
    fk_mat = np.zeros(num_tasks)
    for a in range(1, num_units+1):
        for j in range(0, num_tasks + 1):
            Aaj = "A%03d%03d" % (a, j)
            Aaj_ = m.getVarByName(Aaj)
            # if j is assigned to a, set the value in the matrix
            if Aaj_.X == 1: Aaj_mat[a-1][j] = Aaj_.X
            for k in range(1, num_tasks + 1):
                Xjk = "X%03d%03d" % (j, k)
                Xjk_ = m.getVarByName(Xjk)
                fk = "f%03d" % (k)
                fk_ = m.getVarByName(fk)
                # print(fk_)
                fk_mat[k-1] = fk_.Xn
                # if j comes after k, set the value in the matrix
                if Xjk_.X == 1: Xjk_mat[j][k] = Xjk_.X
    
    # get start time for each task
    task_start_times = np.zeros((num_tasks))
    for k in range(1, num_tasks + 1):
        sk = "s%03d" % (k)
        sk_ = m.getVarByName(sk)
        task_start_times[k-1] = sk_.Xn
        
    # print('\n'.join([' '.join([str(int(Xjk_mat[i][j])) for j in range(num_tasks+1)]) for i in range(num_tasks+1)]))
    schedule = extract_task_assignment(Aaj_mat[0:, 1:], Xjk_mat[1:, 1:], task_start_times)

    if verbose:
        print("Schedule - ", schedule)
        print(obj.getValue())
        m.write("my_model.lp")
        print("\nXjk - \n", Xjk_mat)
        print("\n Aaj - \n", Aaj_mat)
        print("\n fk - \n", fk_mat)
    
    return True, m.Status, obj.getValue(), schedule, duration
    
                    
def extract_task_assignment(matrix_A, matrix_X, task_start_times):
    """ Extract the task assignment from the MILP solution 
    Args:
        matrix_A (np.ndarray): Assignment matrix (set to 1 if task j is assigned to agent a)
        matrix_X (np.ndarray): Sequencing matrix (set to 1 if task j is assigned to agent a before task k)
    Returns:
        list: List of tuples of task and agent assignment
    """
    #print(matrix_A,"\n",  matrix_X)
    agents, tasks = matrix_A.shape
    assignment_list = {}

    # print("sum of data", np.sum(np.array(matrix_X), axis=1))
    for agent in range(agents):
        assigned_tasks = np.where(matrix_A[agent, :] == 1)[0]
        #print(agent, assigned_tasks)
        order = np.sum(np.array(matrix_X[assigned_tasks]), axis=1) 
        # print("matrixX\n", matrix_X[assigned_tasks], order)
        indices = [i[0] for i in sorted(enumerate(order), key=lambda k: k[1], reverse=True)]
        #print(assigned_tasks[indices])
        for index in indices:
            assignment_list[assigned_tasks[index]] = agent
    schedule_order = np.argsort(task_start_times)

    return [(task, assignment_list[task]) for task in schedule_order if task in assignment_list.keys()]


def solve_with_MILP_given_partial_schedule(env, partial_schedule, verbose=False):
    """ Solve the MILP given a partial schedule """
    
    # deconstruct the partial schedule for each agent
    deconstructed_schedule = {}
    for task, agent in partial_schedule:
        if agent not in deconstructed_schedule:
            deconstructed_schedule[agent] = []
        deconstructed_schedule[agent].append(task)

    num_units = env.num_agents
    num_tasks = env.num_tasks
    
    m, mk_ = get_constraints(env, verbose)
    
    # add the partial schedule constraints
    for agent, tasks in deconstructed_schedule.items():
        for i, task in enumerate(tasks):
            # enforce that the task is assigned to the agent
            Aaj = "A%03d%03d" % (agent+1, task+1)
            Aaj_ = m.getVarByName(Aaj)
            m.addConstr(Aaj_ == 1)
            # enforce that the task comes after the 0th task
            X0j_ = m.getVarByName("X000%03d" % (task+1))
            m.addConstr(X0j_ == 1)
            # enforce that the task order for any previous task
            if i > 0:
                Xjk = "X%03d%03d" % (tasks[i-1]+1, task+1)
                Xjk_ = m.getVarByName(Xjk)
                m.addConstr(Xjk_ == 1)
    
    # solve the MILP
    
    # else: 
    m.setObjective(mk_, GRB.MINIMIZE)

    time_start = time.perf_counter()
    m.optimize()
    time_end = time.perf_counter()
    duration = time_end - time_start
    if verbose:
        # print("Status is ####", m.Status)
        status = m.Status
        if status == GRB.UNBOUNDED:
            print('The model cannot be solved because it is unbounded')
            # sys.exit(0)
        if status == GRB.OPTIMAL:
            print('The optimal objective is %g' % m.ObjVal)
            # sys.exit(0)
        if status != GRB.INF_OR_UNBD and status != GRB.INFEASIBLE:
            print('Optimization was stopped with status %d' % status)
            # sys.exit(0)
        if status == GRB.INFEASIBLE:
            # do IIS
            print('The model is infeasible; computing IIS')
            m.computeIIS()
            if m.IISMinimal:
                print('IIS is minimal\n')
            else:
                print('IIS is not minimal\n')
            print('\nThe following constraint(s) cannot be satisfied:')
            for c in m.getConstrs():
                if c.IISConstr:
                    print('%s' % c.ConstrName)
    if m.Status == GRB.INFEASIBLE:
        return False, m.Status, None, None, duration
    
    # build the schedule using assignment map
    obj = m.getObjective()
    Xjk_mat = np.zeros((num_tasks + 1, num_tasks + 1))
    Aaj_mat = np.zeros((num_units, num_tasks + 1))
    fk_mat = np.zeros(num_tasks)
    for a in range(1, num_units+1):
        for j in range(0, num_tasks + 1):
            Aaj = "A%03d%03d" % (a, j)
            Aaj_ = m.getVarByName(Aaj)
            # if j is assigned to a, set the value in the matrix
            if Aaj_.X == 1: Aaj_mat[a-1][j] = Aaj_.X
            for k in range(1, num_tasks + 1):
                Xjk = "X%03d%03d" % (j, k)
                Xjk_ = m.getVarByName(Xjk)
                fk = "f%03d" % (k)
                fk_ = m.getVarByName(fk)
                # print(fk_)
                fk_mat[k-1] = fk_.Xn
                # if j comes after k, set the value in the matrix
                if Xjk_.X == 1: Xjk_mat[j][k] = Xjk_.X
    
    # get start time for each task
    task_start_times = np.zeros((num_tasks))
    for k in range(1, num_tasks + 1):
        sk = "s%03d" % (k)
        sk_ = m.getVarByName(sk)
        task_start_times[k-1] = sk_.Xn
        
    # print('\n'.join([' '.join([str(int(Xjk_mat[i][j])) for j in range(num_tasks+1)]) for i in range(num_tasks+1)]))
    schedule = extract_task_assignment(Aaj_mat[0:, 1:], Xjk_mat[1:, 1:], task_start_times)

    if verbose:
        print("Schedule - ", schedule)
        print(obj.getValue())
        m.write("my_model.lp")
        print("\nXjk - \n", Xjk_mat)
        print("\n Aaj - \n", Aaj_mat)
        print("\n fk - \n", fk_mat)
    
    return True, m.Status, obj.getValue(), schedule, duration
