"""
================================================================================
ADFWI BASELINE (Modified for ICLR 2026 Submission)
--------------------------------------------------------------------------------
This code is based on the ADFWI framework by LiuFeng (SJTU, https://github.com/liufeng2317/ADFWI),
originally released under the MIT License. This version has been modified for ICLR 2026.
Original Author: LiuFeng (SJTU) | Email: liufeng2317@sjtu.edu.cn
================================================================================
"""


import numpy as np
import torch
from torch import Tensor
from typing import Optional,Tuple,Union,List
from uniSI.utils import gpu2cpu,numpy2tensor,tensor2numpy
from uniSI.model.base import AbstractModel
from uniSI.model.parameters import (thomsen_init,elastic_moduli_init,
                         vs_vp_to_Lame,thomsen_to_elastic_moduli,
                         elastic_moduli_for_isotropic,elastic_moduli_for_TI,
                         parameter_staggered_grid)
from uniSI.view import (plot_two_parameter,plot_vp_vs_rho,plot_eps_delta_gamma,plot_lam_mu,plot_model)
from uniSI.survey import Survey

class IsotropicElasticModel(AbstractModel):
    """Isotropic Elastic Velocity model with parameterization of vp vs and rho
    Parameters:
    --------------
        ox (float),oz(float)        : Origin of the model in x- and z- direction (m)
        nx (int),nz(int)            : NUmber of grid points in x- and z- direction (m)
        dx (float),dz(float)        : Grid size in x- and z- direction (m)
        vp (array)                  : P-wave velocity model with shape (nz,nx)
        vs (array)                  : S-wave velocity model with shape (nz,nx)
        rho (array)                 : density model with shape (nz,nx)
        vp_bound (tuple,Optional)   : Bounds for the P-wave velocity model, default None
        vs_bound (tuple,Optional)   : Bounds for the S-wave velocity model, default None
        rho_bound (tuple,Optional)  : Bounds for the density model, default None
        vp_grad (bool,Optional)     : Flag for gradient of S-wave velocity model, default is False
        vs_grad (bool,Optional)     : Flag for gradient of P-wave velocity model, default is False
        rho_grad (bool,Optional)    : Flag for gradient of density, default is False
        free_surface (bool,Optional): Flag for free surface, default is False
        abc_type (str)              : the type of absorbing boundary conditoin : PML,jerjan and other
        abc_jerjan_alpha (float)    : the attenuation factor for jerjan boundary condition
        nabc (int)                  : Number of absorbing boundary cells, default is 20
        device (str,Optional)       : The runing device
        dtype (dtypes,Optional)     : The dtypes for pytorch variable, default is torch.float32
    """
    def __init__(self,
                ox:float,oz:float,
                nx:int,nz:int,
                dx:float,dz:float,
                vp:Optional[Union[np.array,Tensor]]              = None,     # model parameter
                vs:Optional[Union[np.array,Tensor]]              = None,
                rho:Optional[Union[np.array,Tensor]]             = None,
                vp_bound: Optional[Tuple[float, float]]          = None,     # model parameter's boundary
                vs_bound: Optional[Tuple[float, float]]          = None,
                rho_bound: Optional[Tuple[float, float]]         = None,
                vp_grad:Optional[bool]                           = False,    # requires gradient or not
                vs_grad:Optional[bool]                           = False,
                rho_grad:Optional[bool]                          = False,
                free_surface:Optional[bool]                      = False,
                abc_type:Optional[str]                           = 'PML',
                abc_jerjan_alpha:Optional[float]                 = 0.0053,
                nabc:Optional[int]                               = 20,
                auto_update_rho:Optional[bool]                   = True,
                auto_update_vp:Optional[bool]                    = False,
                auto_update_vs:Optional[bool]                    = False,
                water_layer_mask:Optional[Union[np.array,Tensor]]= None,
                device                                           = 'cpu',
                dtype                                            = torch.float32
                )->None:
        # initialize the common model parameters
        super().__init__(ox,oz,nx,nz,dx,dz,free_surface,abc_type,abc_jerjan_alpha,nabc,device,dtype)

        # initialize the thomson model model parameters
        self.pars       = ["vp","vs","rho"]
        self.vp         = vp.copy()
        self.vs         = vs.copy()
        self.rho        = rho.copy()
        self.vp_grad    = vp_grad
        self.vs_grad    = vs_grad
        self.rho_grad   = rho_grad
        self.eps        = np.zeros((nz,nx))
        self.gamma      = np.zeros((nz,nx))
        self.delta      = np.zeros((nz,nx))
        self._parameterization_thomson()
        
        # initialize the lame constant
        self.mu         = None
        self.lamu       = None
        self.lam        = None
        self.muxz       = None
        self.b          = None
        self.bx         = None
        self.bz         = None
        self._parameterization_Lame()
        
        # initialize the elastic moduli
        self.CC         = []
        self._parameterization_elastic_moduli()
        
        # set model bounds
        self.lower_bound["vp"]  = vp_bound[0]  if vp_bound  is not None else None
        self.lower_bound["vs"]  = vs_bound[0]  if vs_bound  is not None else None
        self.lower_bound["rho"] = rho_bound[0] if rho_bound is not None else None
        self.upper_bound["vp"]  = vp_bound[1]  if vp_bound  is not None else None
        self.upper_bound["vs"]  = vs_bound[1]  if vs_bound  is not None else None
        self.upper_bound["rho"] = rho_bound[1] if rho_bound is not None else None
        
        # set model gradients
        self.requires_grad["vp"]    = self.vp_grad
        self.requires_grad["vs"]    = self.vs_grad
        self.requires_grad["rho"]   = self.rho_grad
        
        # check the input model
        self._check_bounds()
        self.check_dims()
        
        # update rho using the empirical function
        self.auto_update_rho = auto_update_rho
        self.auto_update_vp  = auto_update_vp
        self.auto_update_vs  = auto_update_vs
    

        if water_layer_mask is not None:
            self.water_layer_mask = numpy2tensor(water_layer_mask,dtype=torch.bool).to(device)
        else:
            self.water_layer_mask = None
            
    def _parameterization_thomson(self):
        """setting variable and gradients
        """
        # numpy2tensor
        self.vp     = numpy2tensor(self.vp   ,self.dtype).to(self.device)
        self.vs     = numpy2tensor(self.vs   ,self.dtype).to(self.device)
        self.rho    = numpy2tensor(self.rho  ,self.dtype).to(self.device)
        self.eps    = numpy2tensor(self.eps  ,self.dtype).to(self.device)
        self.gamma  = numpy2tensor(self.gamma,self.dtype).to(self.device)
        self.delta  = numpy2tensor(self.delta,self.dtype).to(self.device)
        # set model parameters
        self.vp     = torch.nn.Parameter(self.vp    ,requires_grad=self.vp_grad)
        self.vs     = torch.nn.Parameter(self.vs    ,requires_grad=self.vs_grad)
        self.rho    = torch.nn.Parameter(self.rho   ,requires_grad=self.rho_grad)
        self.eps    = torch.nn.Parameter(self.eps   ,requires_grad=False)
        self.gamma  = torch.nn.Parameter(self.gamma ,requires_grad=False)
        self.delta  = torch.nn.Parameter(self.delta ,requires_grad=False)
        return
    
    def _parameterization_Lame(self):
        """Calculate the lame parameters
        """
        mu,lamu,lam,b   = vs_vp_to_Lame(self.vp,self.vs,self.rho)
        self.mu         = mu
        self.lamu       = lamu
        self.lam        = lam
        self.b          = b
        return
    
    def _parameterization_elastic_moduli(self):
        """calculate the 21 dependent elastic moduli
        """
        # initialize elastic moduli
        CC = [C11,C12,C13,C14,C15,C16,C22,C23,C24,C25,C26,C33,C34,C35,C36,C44,C45,C46,C55,C56,C66] = elastic_moduli_init(self.nz,self.nx,self.device,self.dtype)
        # transform thomsen parameter to elastic moduli 
        C11,C13,C33,C44,C66 = thomsen_to_elastic_moduli(self.vp,self.vs,self.rho,self.eps,self.delta,self.gamma)
        CC = [C11,C12,C13,C14,C15,C16,C22,C23,C24,C25,C26,C33,C34,C35,C36,C44,C45,C46,C55,C56,C66]
        # define elastic moduli for isotropic model
        CC = [C11,C12,C13,C14,C15,C16,C22,C23,C24,C25,C26,C33,C34,C35,C36,C44,C45,C46,C55,C56,C66] = elastic_moduli_for_isotropic(CC)
        # prepare staggered grid settings
        bx,bz,muxz,C44,C55,C66 = parameter_staggered_grid(self.mu,self.b,C44,C55,C66,self.nx,self.nz)
        self.bx = bx
        self.bz = bz
        self.muxz = muxz
        CC = [C11,C12,C13,C14,C15,C16,C22,C23,C24,C25,C26,C33,C34,C35,C36,C44,C45,C46,C55,C56,C66]
        self.CC = CC
        return 
    
    def _plot_vp_vs_rho(self,**kwargs):
        """plot velocity model
        """
        plot_vp_vs_rho(self.vp,self.vs,self.rho,
                            dx=self.dx,dz=self.dz,**kwargs)
        return
    
    def _plot_eps_delta_gamma(self,**kwargs):
        """plot anisotropic parameters
        """
        plot_eps_delta_gamma(self.eps,self.delta,self.gamma,
                            dx=self.dx,dz=self.dz,**kwargs)
        return
    
    def _plot_lam_mu(self,**kwargs):
        """plot lame parameters
        """
        plot_lam_mu(self.lam,self.mu,
                            dx=self.dx,dz=self.dz,**kwargs)
        return
    
    def _plot(self,var,**kwargs):
        """plot single velocity model
        """
        model_data = self.get_model(var)
        plot_model(model_data,title=var,**kwargs)
        return
    
    def set_rho_using_empirical_function(self):
        """approximate rho via empirical relations with vp
        """
        rho         = self.rho.cpu().detach().numpy()
        vp          = self.vp.cpu().detach().numpy()
        rho_empirical  = np.power(vp, 0.25) * 310
        if self.water_layer_mask is not None:
            mask = self.water_layer_mask.cpu().detach().numpy()
            rho_empirical[mask] = rho[mask]
        rho         = numpy2tensor(rho_empirical,self.dtype).to(self.device)
        self.rho    = torch.nn.Parameter(rho   ,requires_grad=self.rho_grad)
        return
    
    def set_vp_using_empirical_function(self):
        """approximate vp via empirical relations with vs
        """
        vp = self.vp.cpu().detach().numpy()
        vs = self.vs.cpu().detach().numpy()
        vp_empirical = vs*np.sqrt(3)
        if self.water_layer_mask is not None:
            mask = self.water_layer_mask.cpu().detach().numpy()
            vp_empirical[mask] = vp[mask]
        vp = numpy2tensor(vp_empirical,self.dtype).to(self.device)
        self.vp = torch.nn.Parameter(vp,requires_grad=self.vp_grad)
        return
    
    def set_vs_using_empirical_function(self):
        """approximate vs via empirical relations with vp"""
        vp = self.vp.cpu().detach().numpy()
        vs = self.vs.cpu().detach().numpy()
        vs_empirical = vp / np.sqrt(3)
        if self.water_layer_mask is not None:
            mask = self.water_layer_mask.cpu().detach().numpy()
            vs_empirical[mask] = vs[mask]
        vs = numpy2tensor(vs_empirical, self.dtype).to(self.device)
        self.vs = torch.nn.Parameter(vs, requires_grad=self.vs_grad)
        return
    
    def clip_params(self)->None:
        """Clip the model parameters to the given bounds
        """
        for par in self.pars:
            if self.lower_bound[par] is not None and self.upper_bound[par] is not None:
                # Retrieve the model parameter
                m = getattr(self, par)
                min_value = self.lower_bound[par]
                max_value = self.upper_bound[par]

                # Create a temporary copy for masking purposes
                m_temp = m.clone()  # Use .clone() instead of .copy() to avoid issues with gradients

                # Clip the values of the parameter using in-place modification with .data
                m.data.clamp_(min_value, max_value)

                # Apply the water layer mask if it is not None, using in-place modification
                if self.water_layer_mask is not None:
                    m.data = torch.where(self.water_layer_mask, m_temp.data, m.data)
        return
    

    def forward(self) -> None:
        """Forward method of the elastic model class
        
        """
        # set the constraints on the parameters if necessary
        if self.auto_update_rho:
            self.set_rho_using_empirical_function()
            
        if self.auto_update_vp:
            self.set_vp_using_empirical_function()

        if self.auto_update_vs:
            self.set_vs_using_empirical_function()
        
        # Clip the model parameters
        self.clip_params()
        
        # calculate the thomson/lame and elastic moduli parameters
        self._parameterization_Lame()
        self._parameterization_elastic_moduli()
        return 
    
    