import torch

class EarlyStopping:
    def __init__(self, patience=5, mode="max", delta=0.0, save_path="best_model.pth"):
        """
        Args:
        - patience (int): Number of epochs with no improvement after which training will be stopped.
        - mode (str): "max" for maximizing the metric, "min" for minimizing.
        - delta (float): Minimum change in the monitored metric to qualify as an improvement.
        - save_path (str): Path to save the best model.
        """
        self.patience = patience
        self.mode = mode
        self.delta = delta
        self.save_path = save_path
        self.best_score = None
        self.counter = 0
        self.early_stop = False

    def __call__(self, score, model):
        """
        Check if training should stop based on the given score.
        Save the model if the score improves.
        """
        if self.best_score is None:
            self.best_score = score
            self.save_checkpoint(model)
        elif (self.mode == "max" and score <= self.best_score + self.delta) or \
             (self.mode == "min" and score >= self.best_score - self.delta):
            self.counter += 1
            if self.counter >= self.patience:
                self.early_stop = True
        else:
            self.best_score = score
            self.counter = 0
            self.save_checkpoint(model)

    def save_checkpoint(self, model):
        """
        Save the current best model to the specified path.
        """
        print(f"Saving model with score {self.best_score} to {self.save_path}")
        torch.save(model.state_dict(), self.save_path)
