'''CIFAR10, GCNN/ResNet18 Training Code.'''
from __future__ import print_function

import torch
import torch.nn as nn
import torch.optim as optim
import torch.nn.functional as F
import torch.backends.cudnn as cudnn
from torch.optim.lr_scheduler import MultiStepLR

import torchvision
import torchvision.transforms as transforms

import os
import argparse

from models.resnet import *
from utils import progress_bar
from torch.autograd import Variable
import random

parser = argparse.ArgumentParser(description='PyTorch CIFAR10 Training')
parser.add_argument('--lr', default=0.1, type=float, help='learning rate')
parser.add_argument('--n_epochs', default=120, type=int)
parser.add_argument('--train_angle', default=0, type=int)
parser.add_argument('--test_angle', default=0, type=int)
parser.add_argument('--translate', default=0.0, type=float)
parser.add_argument('--translate_by', default=0.0, type=float)
args = parser.parse_args()

use_cuda = torch.cuda.is_available()
best_acc = 0  # best test accuracy
start_epoch = 0  # start from epoch 0 or last checkpoint epoch
n_epochs = args.n_epochs
train_angle = args.translate*180
test_angle = 0
translate = args.translate
translate_by = args.translate_by


# Data
means = (0.4914, 0.4822, 0.4465)
print('==> Preparing data..')
transform_train = transforms.Compose([
    #transforms.RandomCrop(32, padding=4),
    #transforms.RandomHorizontalFlip(),
    transforms.Lambda(lambda image: image.rotate(random.random() * train_angle * 2 - train_angle)),
    transforms.RandomAffine(degrees=0, translate=(translate_by, translate_by)),
    transforms.ToTensor(),
    transforms.Normalize(means, (0.2023, 0.1994, 0.2010)),
])

transform_test = transforms.Compose([
    transforms.ToTensor(),
    transforms.Normalize(means, (0.2023, 0.1994, 0.2010)),
])

trainset = torchvision.datasets.CIFAR10(root='../data', train=True, download=True, transform=transform_train)
trainloader = torch.utils.data.DataLoader(trainset, batch_size=128, shuffle=True, num_workers=2)

testset = torchvision.datasets.CIFAR10(root='../data', train=False, download=True, transform=transform_test)
testloader = torch.utils.data.DataLoader(testset, batch_size=100, shuffle=False, num_workers=2)

classes = ('plane', 'car', 'bird', 'cat', 'deer', 'dog', 'frog', 'horse', 'ship', 'truck')

# Model
#if args.resume:
#    # Load checkpoint.
#    print('==> Resuming from checkpoint..')
#    assert os.path.isdir(args.checkpoint), 'Error: no checkpoint directory found!'
#    checkpoint = torch.load(os.path.join(args.checkpoint, 'ckpt.t7'))
#    net = checkpoint['net']
#    best_acc = checkpoint['acc']
#    start_epoch = checkpoint['epoch']
#else:
print('==> Building model..')
net = ResNet18()

if use_cuda:
    net.cuda()
    net = torch.nn.DataParallel(net, device_ids=range(torch.cuda.device_count()))
    cudnn.benchmark = True

criterion = nn.CrossEntropyLoss()
optimizer = optim.SGD(net.parameters(), lr=args.lr, momentum=0.9, weight_decay=5e-4)

# Training
def train(epoch):
    print('\nEpoch: %d' % epoch)
    net.train()
    train_loss = 0
    correct = 0
    total = 0
    for batch_idx, (inputs, targets) in enumerate(trainloader):
        if use_cuda:
            inputs, targets = inputs.cuda(), targets.cuda()
        optimizer.zero_grad()
        inputs, targets = Variable(inputs), Variable(targets)
        outputs = net(inputs)
        loss = criterion(outputs, targets)
        loss.backward()
        optimizer.step()

        train_loss += loss.data[0]
        _, predicted = torch.max(outputs.data, 1)
        total += targets.size(0)
        correct += predicted.eq(targets.data).cpu().sum()

        progress_bar(batch_idx, len(trainloader), 'Loss: %.3f | Acc: %.3f%% (%d/%d)'
            % (train_loss/(batch_idx+1), 100.*correct/total, correct, total))

def test(epoch):
    global best_acc
    net.eval()
    test_loss = 0
    correct = 0
    total = 0
    for batch_idx, (inputs, targets) in enumerate(testloader):
        if use_cuda:
            inputs, targets = inputs.cuda(), targets.cuda()
        inputs, targets = Variable(inputs, volatile=True), Variable(targets)
        outputs = net(inputs)
        loss = criterion(outputs, targets)

        test_loss += loss.data[0]
        _, predicted = torch.max(outputs.data, 1)
        total += targets.size(0)
        correct += predicted.eq(targets.data).cpu().sum()

        progress_bar(batch_idx, len(testloader), 'Loss: %.3f | Acc: %.3f%% (%d/%d)'
            % (test_loss/(batch_idx+1), 100.*correct/total, correct, total))

    # Save checkpoint.
    #acc = 100.*correct/total
    #if acc > best_acc:
    #    print('Saving..')
    #    state = {
    #        'net': net.module if use_cuda else net,
    #        'acc': acc,
    #        'epoch': epoch,
    #    }
    #    if not os.path.isdir(args.checkpoint):
    #        os.mkdir(args.checkpoint)
    #    torch.save(state, os.path.join(args.checkpoint, 'ckpt.t7'))
    #    best_acc = acc

milestones = [50, 100, 140]
scheduler = MultiStepLR(optimizer, milestones, gamma=0.1)

for epoch in range(start_epoch, n_epochs):
    train(epoch)
    scheduler.step()
    torch.save(net, "./saved_models/gcnn_resnet18_cifar10_"+str(translate)+"_trans_rot.pth")
    test(epoch)

