# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.

# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.
# --------------------------------------------------------
# References:
# DeiT: https://github.com/facebookresearch/deit
# BEiT: https://github.com/microsoft/unilm/tree/master/beit
# --------------------------------------------------------

import argparse
import datetime
import json
import numpy as np
import os
import time
from pathlib import Path

import torch
import torch.backends.cudnn as cudnn
from torch.utils.tensorboard import SummaryWriter
from torchvision import transforms, datasets

import timm

assert timm.__version__ == "0.3.2" # version check
from timm.models.layers import trunc_normal_
from timm.data.mixup import Mixup
from timm.loss import LabelSmoothingCrossEntropy, SoftTargetCrossEntropy
from timm.utils import accuracy

import util.lr_decay as lrd
import util.misc as misc
from util.datasets import build_dataset, CustomDataset, CustomDataset_selected, CustomDataset_ImgList, build_transform
from util.pos_embed import interpolate_pos_embed
from util.misc import NativeScalerWithGradNormCount as NativeScaler

import models_vit
import models_mae
import engine_pretrain

import MOOD.MOODv1.ood_utils as ood_utils

from openood.pipelines.test_ood_pipeline import TestOODPipeline
from openood.utils.config import Config, merge_configs

# from engine_finetune import train_one_epoch, evaluate


def get_args_parser():
    parser = argparse.ArgumentParser('MAE fine-tuning for image classification', add_help=False)
    parser.add_argument('--batch_size', default=64, type=int,
                        help='Batch size per GPU (effective batch size is batch_size * accum_iter * # gpus')
    parser.add_argument('--epochs', default=50, type=int)
    parser.add_argument('--accum_iter', default=1, type=int,
                        help='Accumulate gradient iterations (for increasing the effective batch size under memory constraints)')

    # Model parameters
    parser.add_argument('--model', default='mae_vit_base_patch16', type=str, metavar='MODEL',
                        help='Name of model to train')

    parser.add_argument('--input_size', default=224, type=int,
                        help='images input size')

    parser.add_argument('--drop_path', type=float, default=0.1, metavar='PCT',
                        help='Drop path rate (default: 0.1)')

    parser.add_argument('--mask_ratio', default=0.75, type=float,
                        help='Masking ratio (percentage of removed patches).')

    parser.add_argument('--norm_pix_loss', action='store_true',
                        help='Use (per-patch) normalized pixels as targets for computing loss')
    parser.set_defaults(norm_pix_loss=True) # JY

    # Optimizer parameters
    parser.add_argument('--clip_grad', type=float, default=None, metavar='NORM',
                        help='Clip gradient norm (default: None, no clipping)')
    parser.add_argument('--weight_decay', type=float, default=0.05,
                        help='weight decay (default: 0.05)')

    parser.add_argument('--lr', type=float, default=None, metavar='LR',
                        help='learning rate (absolute lr)')
    parser.add_argument('--blr', type=float, default=5e-4, metavar='LR',
                        help='base learning rate: absolute_lr = base_lr * total_batch_size / 256')
    parser.add_argument('--layer_decay', type=float, default=0.75,
                        help='layer-wise lr decay from ELECTRA/BEiT')

    parser.add_argument('--min_lr', type=float, default=1e-6, metavar='LR',
                        help='lower lr bound for cyclic schedulers that hit 0')

    parser.add_argument('--warmup_epochs', type=int, default=5, metavar='N',
                        help='epochs to warmup LR')

    # Augmentation parameters
    parser.add_argument('--color_jitter', type=float, default=None, metavar='PCT',
                        help='Color jitter factor (enabled only when not using Auto/RandAug)')
    parser.add_argument('--aa', type=str, default='rand-m9-mstd0.5-inc1', metavar='NAME',
                        help='Use AutoAugment policy. "v0" or "original". " + "(default: rand-m9-mstd0.5-inc1)'),
    parser.add_argument('--smoothing', type=float, default=0.1,
                        help='Label smoothing (default: 0.1)')

    # * Random Erase params
    parser.add_argument('--reprob', type=float, default=0.25, metavar='PCT',
                        help='Random erase prob (default: 0.25)')
    parser.add_argument('--remode', type=str, default='pixel',
                        help='Random erase mode (default: "pixel")')
    parser.add_argument('--recount', type=int, default=1,
                        help='Random erase count (default: 1)')
    parser.add_argument('--resplit', action='store_true', default=False,
                        help='Do not random erase first (clean) augmentation split')

    # * Mixup params
    parser.add_argument('--mixup', type=float, default=0, # JY
                        help='mixup alpha, mixup enabled if > 0.')
    parser.add_argument('--cutmix', type=float, default=0, # JY
                        help='cutmix alpha, cutmix enabled if > 0.')
    parser.add_argument('--cutmix_minmax', type=float, nargs='+', default=None,
                        help='cutmix min/max ratio, overrides alpha and enables cutmix if set (default: None)')
    parser.add_argument('--mixup_prob', type=float, default=1.0,
                        help='Probability of performing mixup or cutmix when either/both is enabled')
    parser.add_argument('--mixup_switch_prob', type=float, default=0.5,
                        help='Probability of switching to cutmix when both mixup and cutmix enabled')
    parser.add_argument('--mixup_mode', type=str, default='batch',
                        help='How to apply mixup/cutmix params. Per "batch", "pair", or "elem"')

    # * Finetuning params
    parser.add_argument('--finetune', default='', # JY
                        help='finetune from checkpoint')
    parser.add_argument('--global_pool', action='store_true')
    parser.set_defaults(global_pool=True)
    parser.add_argument('--cls_token', action='store_false', dest='global_pool',
                        help='Use class token instead of global pool for classification')

    # Dataset parameters
    parser.add_argument('--data_path', default='/data3/jiangy/project/openood/OpenOOD-main/data/images_classic/cifar10/', type=str,
                        help='dataset path')
    parser.add_argument('--nb_classes', default=10, type=int,
                        help='number of the classification types')

    parser.add_argument('--output_dir', default='./output_dir/test1',
                        help='path where to save, empty for no saving')
    parser.add_argument('--log_dir', default='./output_dir/test1',
                        help='path where to tensorboard log')
    parser.add_argument('--device', default='cuda:0',
                        help='device to use for training / testing')
    parser.add_argument('--seed', default=0, type=int)
    parser.add_argument('--resume', default='/data3/jiangy/project/openood/test/output_dir/cifar10_longtail_0/checkpoint-99.pth',
                        help='resume from checkpoint')

    parser.add_argument('--start_epoch', default=0, type=int, metavar='N',
                        help='start epoch')
    parser.add_argument('--eval', action='store_true',
                        help='Perform evaluation only')
    parser.set_defaults(eval=True) # JY
    parser.add_argument('--dist_eval', action='store_true', default=False,
                        help='Enabling distributed evaluation (recommended during training for faster monitor')
    parser.add_argument('--num_workers', default=4, type=int)
    parser.add_argument('--pin_mem', action='store_true',
                        help='Pin CPU memory in DataLoader for more efficient (sometimes) transfer to GPU.')
    parser.add_argument('--no_pin_mem', action='store_false', dest='pin_mem')
    parser.set_defaults(pin_mem=True)

    # distributed training parameters
    parser.add_argument('--world_size', default=1, type=int,
                        help='number of distributed processes')
    parser.add_argument('--local_rank', default=-1, type=int)
    parser.add_argument('--dist_on_itp', action='store_true')
    parser.add_argument('--dist_url', default='env://',
                        help='url used to set up distributed training')
    
    # new parameters
    parser.add_argument('--ds_type', default='self-supervised', type=str)
    parser.add_argument('--ood_dataset', default='cifar10', type=str)
    parser.add_argument('--pre', default='ft', type=str)
    parser.add_argument('--test_folder', default='', type=str)
    parser.add_argument('--is_gray', default=False, type=bool)
    parser.add_argument('--longtail_idx', default=-1, type=int)
    parser.add_argument('--ft_mask', action='store_true', default=False)
    return parser


def main(args):
    misc.init_distributed_mode(args)

    print('job dir: {}'.format(os.path.dirname(os.path.realpath(__file__))))
    print("{}".format(args).replace(', ', ',\n'))

    device = torch.device(args.device)
    torch.cuda.set_device(device)

    # fix the seed for reproducibility
    seed = args.seed + misc.get_rank()
    torch.manual_seed(seed)
    np.random.seed(seed)

    cudnn.benchmark = True
    args.ood_data = True
    # dataset_train = build_dataset(is_train=True, args=args)
    # dataset_val = build_dataset(is_train=False, args=args)
    dataset_train = CustomDataset_ImgList(root='/data3/jiangy/project/openood/OpenOOD-main/data/images_classic', file_path='/data3/jiangy/project/openood/OpenOOD-main/data/benchmark_imglist/{}/train_{}.txt'.format('mnist', 'mnist'), transform=build_transform(True, args, False))
    dataset_val = CustomDataset_ImgList(root='/data3/jiangy/project/openood/OpenOOD-main/data/images_classic', file_path='/data3/jiangy/project/openood/OpenOOD-main/data/benchmark_imglist/{}/test_{}.txt'.format('mnist', 'mnist'), transform=build_transform(False, args, False))
    
    transform_ood = transforms.Compose([
            transforms.RandomResizedCrop(args.input_size, scale=(0.2, 1.0), interpolation=3),  # 3 is bicubic
            transforms.RandomHorizontalFlip(),
            transforms.ToTensor(),
            transforms.Normalize(mean=[0.485, 0.456, 0.406], std=[0.229, 0.224, 0.225])])
    # dataset_ood = datasets.ImageFolder(os.path.join('/data3/jiangy/project/openood/OpenOOD-main/data/images_classic/svhn', 'test'), transform=transform_ood)
    if args.ood_dataset not in ['places365', 'texture'] and args.test_folder == '':
        args.test_folder = 'test'
    if args.ood_dataset in ['openimage_o']:
        dataset_ood = datasets.ImageFolder('/data3/jiangy/project/openood/OpenOOD-main/data/images_largescale/openimage_o/', transform=transform_ood)
    elif args.ood_dataset == 'svhn':
        dataset_ood = CustomDataset_selected(root='/data3/jiangy/project/openood/OpenOOD-main/data/images_classic/svhn/test', transform=transform_ood)
    elif '_p' in args.ood_dataset:
        dataset_ood = datasets.ImageFolder('/data3/jiangy/project/openood/OpenOOD-main/data/images_classic/{}/{}'.format(args.ood_dataset, args.test_folder), transform=transform_ood)
    # elif args.ds_type == 'color_change':
    #     dataset_ood = CustomDataset(root='/data3/jiangy/project/openood/OpenOOD-main/data/images_classic/cifar10/test', transform=transform_ood, labels_include=[9])
    else:
        dataset_ood = datasets.ImageFolder('/data3/jiangy/project/openood/OpenOOD-main/data/images_classic/{}/{}'.format(args.ood_dataset, args.test_folder), transform=transform_ood)
    if args.ds_type.startswith('color_change'):
        dataset_ood = CustomDataset(root='/data3/jiangy/project/openood/OpenOOD-main/data/images_classic/{}/{}'.format(args.ood_dataset, args.test_folder), transform=transform_ood, labels_include=[int(args.ds_type[-1])])
    print(dataset_ood)
    print(len(dataset_ood))

    if True:  # args.distributed:
        num_tasks = misc.get_world_size()
        global_rank = misc.get_rank()
        sampler_train = torch.utils.data.DistributedSampler(
            dataset_train, num_replicas=num_tasks, rank=global_rank, shuffle=True
        )
        print("Sampler_train = %s" % str(sampler_train))
        # sampler_ss = torch.utils.data.DistributedSampler(
        #     dataset_ss, num_replicas=num_tasks, rank=global_rank, shuffle=True
        # )
        # print("Sampler_ss = %s" % str(sampler_ss))
        if args.dist_eval:
            if len(dataset_val) % num_tasks != 0:
                print('Warning: Enabling distributed evaluation with an eval dataset not divisible by process number. '
                      'This will slightly alter validation results as extra duplicate entries are added to achieve '
                      'equal num of samples per-process.')
            sampler_val = torch.utils.data.DistributedSampler(
                dataset_val, num_replicas=num_tasks, rank=global_rank, shuffle=True)  # shuffle=True to reduce monitor bias
        else:
            sampler_val = torch.utils.data.SequentialSampler(dataset_val)
            sampler_ood = torch.utils.data.SequentialSampler(dataset_ood)
    else:
        sampler_train = torch.utils.data.RandomSampler(dataset_train)
        sampler_val = torch.utils.data.SequentialSampler(dataset_val)
        sampler_ss = torch.utils.data.RandomSampler(dataset_ss)

    if global_rank == 0 and args.log_dir is not None and not args.eval:
        os.makedirs(args.log_dir, exist_ok=True)
        log_writer = SummaryWriter(log_dir=args.log_dir)
    else:
        log_writer = None

    data_loader_train = torch.utils.data.DataLoader(
        dataset_train, sampler=sampler_train,
        batch_size=args.batch_size,
        num_workers=args.num_workers,
        pin_memory=args.pin_mem,
        drop_last=True,
    )

    data_loader_val = torch.utils.data.DataLoader(
        dataset_val, sampler=sampler_val,
        batch_size=args.batch_size,
        num_workers=args.num_workers,
        pin_memory=args.pin_mem,
        drop_last=False
    )

    data_loader_ood = torch.utils.data.DataLoader(
        dataset_ood, sampler=sampler_ood,
        batch_size=args.batch_size,
        num_workers=args.num_workers,
        pin_memory=args.pin_mem,
        drop_last=False
    )
    print(len(data_loader_ood))

    mixup_fn = None
    mixup_active = args.mixup > 0 or args.cutmix > 0. or args.cutmix_minmax is not None
    if mixup_active:
        print("Mixup is activated!")
        mixup_fn = Mixup(
            mixup_alpha=args.mixup, cutmix_alpha=args.cutmix, cutmix_minmax=args.cutmix_minmax,
            prob=args.mixup_prob, switch_prob=args.mixup_switch_prob, mode=args.mixup_mode,
            label_smoothing=args.smoothing, num_classes=args.nb_classes)
    
    # model = models_vit.__dict__[args.model](
    #     num_classes=args.nb_classes,
    #     drop_path_rate=args.drop_path,
    #     global_pool=args.global_pool,
    # )
    model = models_mae.__dict__[args.model](norm_pix_loss=args.norm_pix_loss)
    model.num_classes = args.nb_classes
    model.head = torch.nn.Linear(model.embed_dim, model.num_classes).to(device) #torch.nn.Sequential(torch.nn.BatchNorm1d(model.embed_dim, affine=False, eps=1e-6), torch.nn.Linear(model.embed_dim, model.num_classes)).to(device)
    model.set_ft_mask(args.ft_mask, args.mask_ratio)

    if args.finetune and not args.eval:
        checkpoint = torch.load(args.finetune, map_location='cpu')

        print("Load pre-trained checkpoint from: %s" % args.finetune)
        checkpoint_model = checkpoint['model']
        state_dict = model.state_dict()
        for k in ['head.weight', 'head.bias']:
            if k in checkpoint_model and checkpoint_model[k].shape != state_dict[k].shape:
                print(f"Removing key {k} from pretrained checkpoint")
                del checkpoint_model[k]

        # interpolate position embedding
        interpolate_pos_embed(model, checkpoint_model)

        # load pre-trained model
        msg = model.load_state_dict(checkpoint_model, strict=False)
        print(msg)

        if args.global_pool:
            assert set(msg.missing_keys) == {'head.weight', 'head.bias', 'fc_norm.weight', 'fc_norm.bias'}
        else:
            assert set(msg.missing_keys) == {'head.weight', 'head.bias'}

        # manually initialize fc layer
        trunc_normal_(model.head.weight, std=2e-5)

    model.to(device)

    model_without_ddp = model
    n_parameters = sum(p.numel() for p in model.parameters() if p.requires_grad)

    # print("Model = %s" % str(model_without_ddp))
    print('number of params (M): %.2f' % (n_parameters / 1.e6))

    eff_batch_size = args.batch_size * args.accum_iter * misc.get_world_size()
    
    if args.lr is None:  # only base_lr is specified
        args.lr = args.blr * eff_batch_size / 256

    print("base lr: %.2e" % (args.lr * 256 / eff_batch_size))
    print("actual lr: %.2e" % args.lr)

    print("accumulate grad iterations: %d" % args.accum_iter)
    print("effective batch size: %d" % eff_batch_size)

    if args.distributed:
        model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.gpu])
        model_without_ddp = model.module

    # build optimizer with layer-wise lr decay (lrd)
    param_groups = lrd.param_groups_lrd(model_without_ddp, args.weight_decay,
        no_weight_decay_list=model_without_ddp.no_weight_decay(),
        layer_decay=args.layer_decay
    )
    optimizer = torch.optim.AdamW(param_groups, lr=args.lr)
    loss_scaler = NativeScaler()

    if mixup_fn is not None:
        # smoothing is handled with mixup label transform
        criterion = SoftTargetCrossEntropy()
    elif args.smoothing > 0.:
        criterion = LabelSmoothingCrossEntropy(smoothing=args.smoothing)
    else:
        criterion = torch.nn.CrossEntropyLoss()

    print("criterion = %s" % str(criterion))
    
    if args.resume.endswith('bma.pth'):
        chkpt = torch.load(args.resume, map_location='cpu')
        model_without_ddp.load_state_dict(chkpt, strict=False)
        print('=' * 20)
        print('bma model')
    else:
        misc.load_model(args=args, model_without_ddp=model_without_ddp, optimizer=optimizer, loss_scaler=loss_scaler)

    # _, feat_train, label_train, pred_train = evaluate(data_loader_train, model, device)

    config = setup_config(args=args)
    pipeline = TestOODPipeline(config)
    pipeline.run(model)
    

    # with open(os.path.join(args.output_dir, 'log_ood{}.txt'.format(args.ood_dataset)), 'a') as f:
    #     # f.write(datetime.l)
    #     f.write('=' * 10)
    #     f.write(str(args))
    #     f.write('\n')
    #     f.write('fpr95: {}, auroc: {}, aupr: {}'.format(fpr95, auroc, aupr))
    #     f.write('\n')


def setup_config(config_process_order=('merge', 'parse_args', 'parse_refs'), args=None):
    id = 'cifar10'
    if 'cifar100' in args.output_dir and 'ssl_cifar100' not in args.output_dir:
        id = 'cifar100'
    elif 'covid' in args.output_dir:
        id = 'covid'
    elif 'imagenet200' in args.output_dir:
        id = 'imagenet200'
    oodtype = 'ood'
    if 'recall' in args.pre:
        oodtype = 'hidden'
    parser = argparse.ArgumentParser()
    parser.add_argument('--config', dest='config', nargs='+', required=True)
    opt, unknown_args = parser.parse_known_args(['--config', '../OpenOOD-main/configs/datasets/{}/{}.yml'.format(id, id),
    '../OpenOOD-main/configs/datasets/{}/{}_{}.yml'.format(id, id, oodtype),
    '../OpenOOD-main/configs/networks/vit.yml',
    '../OpenOOD-main/configs/pipelines/test/test_ood.yml', 
    '../OpenOOD-main/configs/preprocessors/base_preprocessor.yml',
    '../OpenOOD-main/configs/postprocessors/mds.yml',
    '--num_workers', '8',
    '--dataset.pre_size', '224',
    '--dataset.image_size', '224',
    '--dataset.train.batch_size', '256',
    # '--dataset.longtail', '{}'.format(args.longtail_idx),
    '--output_dir', '{}_{}'.format(args.output_dir, args.pre),
    '--seed', '1',
    '--network.checkpoint', '/home/jiangy/project/newdata_project/ood/OpenOOD-main/results/cifar10_vit-b-16_base_e100_lr0.0005_default/s0/best.ckpt'])
    config = [Config(path) for path in opt.config]

    for process in config_process_order:
        if process == 'merge':
            config = merge_configs(*config)
        elif process == 'parse_args':
            if isinstance(config, Config):
                config.parse_args(unknown_args)
            else:
                for cfg in config:
                    cfg.parse_args(unknown_args)
        elif process == 'parse_refs':
            if isinstance(config, Config):
                config.parse_refs()
            else:
                for cfg in config:
                    cfg.parse_refs()
        else:
            raise ValueError('unknown config process name: {}'.format(process))

    # manually modify 'output_dir'
    config.output_dir = os.path.join(config.output_dir, config.exp_name)

    return config


@torch.no_grad()
def evaluate(data_loader, model, device, ood=False):
    criterion = torch.nn.CrossEntropyLoss()

    metric_logger = misc.MetricLogger(delimiter="  ")
    header = 'Test:' if not ood else 'Test (ood):'

    feat_list = []
    label_list = []
    output_list = []
    pred_list = []

    # switch to evaluation mode
    model.eval()

    for batch in metric_logger.log_every(data_loader, 10, header):
        images = batch[0]
        target = batch[-1]
        images = images.to(device, non_blocking=True)
        target = target.to(device, non_blocking=True)

        # compute output
        with torch.cuda.amp.autocast():
            # output = model(images)
            model(images)
            feat = model.feat
            if not ood:
                output = model.head(feat)
                loss = criterion(output, target)

        feat_list.append(feat.cpu().detach())
        label_list.append(target.cpu().detach())
        pred_list.append(torch.argmax(output).cpu().detach())

        if not ood:
            acc1, acc5 = accuracy(output, target, topk=(1, 5 if model.num_classes > 5 else model.num_classes))
            output_list.append(output.cpu().detach().numpy())

            batch_size = images.shape[0]
            metric_logger.update(loss=loss.item())
            metric_logger.meters['acc1'].update(acc1.item(), n=batch_size)
            metric_logger.meters['acc5'].update(acc5.item(), n=batch_size)
    # gather the stats from all processes
    metric_logger.synchronize_between_processes()
    if not ood:
        print('* Acc@1 {top1.global_avg:.3f} Acc@5 {top5.global_avg:.3f} loss {losses.global_avg:.3f}'
            .format(top1=metric_logger.acc1, top5=metric_logger.acc5, losses=metric_logger.loss))

    return {k: meter.global_avg for k, meter in metric_logger.meters.items()}, torch.cat(feat_list), torch.cat(label_list), torch.cat(pred_list)



if __name__ == '__main__':
    args = get_args_parser()
    args = args.parse_args()
    if args.output_dir:
        Path(args.output_dir).mkdir(parents=True, exist_ok=True)
    main(args)
