# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.

# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.
# --------------------------------------------------------
# References:
# DeiT: https://github.com/facebookresearch/deit
# BEiT: https://github.com/microsoft/unilm/tree/master/beit
# --------------------------------------------------------

import argparse
import datetime
import json
import numpy as np
import os
import time
from pathlib import Path

import torch
import torch.backends.cudnn as cudnn
from torch.utils.tensorboard import SummaryWriter
from torchvision import datasets

import scipy as sp

import timm

assert timm.__version__ == "0.3.2" # version check
import timm.optim.optim_factory as optim_factory
from timm.models.layers import trunc_normal_
from timm.data.mixup import Mixup
from timm.loss import LabelSmoothingCrossEntropy, SoftTargetCrossEntropy

import util.lr_decay as lrd
import util.misc as misc
import util.datasets
from util.datasets import build_dataset, build_transform, CustomDataset_ImgList
from util.pos_embed import interpolate_pos_embed
from util.misc import NativeScalerWithGradNormCount as NativeScaler

import models_vit
import models_mae
import engine_pretrain

from engine_finetune import train_one_epoch, evaluate, evaluate_new, train_one_epoch_ss, train_one_epoch_ss_parallel, train_one_epoch_ss_seq_parallel, GeneralMovingAverage

import warnings
warnings.filterwarnings('ignore')


def get_args_parser():
    parser = argparse.ArgumentParser('MAE fine-tuning for image classification', add_help=False)
    parser.add_argument('--batch_size', default=64, type=int,
                        help='Batch size per GPU (effective batch size is batch_size * accum_iter * # gpus')
    parser.add_argument('--epochs', default=100, type=int)
    parser.add_argument('--accum_iter', default=1, type=int,
                        help='Accumulate gradient iterations (for increasing the effective batch size under memory constraints)')

    # Model parameters
    parser.add_argument('--model', default='mae_vit_base_patch16', type=str, metavar='MODEL',
                        help='Name of model to train')

    parser.add_argument('--input_size', default=224, type=int,
                        help='images input size')

    parser.add_argument('--drop_path', type=float, default=0.1, metavar='PCT',
                        help='Drop path rate (default: 0.1)')

    parser.add_argument('--mask_ratio', default=0.75, type=float,
                        help='Masking ratio (percentage of removed patches).')

    parser.add_argument('--norm_pix_loss', action='store_true',
                        help='Use (per-patch) normalized pixels as targets for computing loss')
    parser.set_defaults(norm_pix_loss=True) # JY

    # Optimizer parameters
    parser.add_argument('--clip_grad', type=float, default=None, metavar='NORM',
                        help='Clip gradient norm (default: None, no clipping)')
    parser.add_argument('--weight_decay', type=float, default=0.05,
                        help='weight decay (default: 0.05)')

    parser.add_argument('--lr', type=float, default=None, metavar='LR',
                        help='learning rate (absolute lr)')
    parser.add_argument('--blr', type=float, default=1.5e-4, metavar='LR',
                        help='base learning rate: absolute_lr = base_lr * total_batch_size / 256')
    parser.add_argument('--layer_decay', type=float, default=0.75,
                        help='layer-wise lr decay from ELECTRA/BEiT')

    parser.add_argument('--min_lr', type=float, default=1e-6, metavar='LR',
                        help='lower lr bound for cyclic schedulers that hit 0')

    parser.add_argument('--warmup_epochs', type=int, default=5, metavar='N',
                        help='epochs to warmup LR')

    # Augmentation parameters
    parser.add_argument('--color_jitter', type=float, default=None, metavar='PCT',
                        help='Color jitter factor (enabled only when not using Auto/RandAug)')
    parser.add_argument('--aa', type=str, default='rand-m9-mstd0.5-inc1', metavar='NAME',
                        help='Use AutoAugment policy. "v0" or "original". " + "(default: rand-m9-mstd0.5-inc1)'),
    parser.add_argument('--smoothing', type=float, default=0.1,
                        help='Label smoothing (default: 0.1)')

    # * Random Erase params
    parser.add_argument('--reprob', type=float, default=0.25, metavar='PCT',
                        help='Random erase prob (default: 0.25)')
    parser.add_argument('--remode', type=str, default='pixel',
                        help='Random erase mode (default: "pixel")')
    parser.add_argument('--recount', type=int, default=1,
                        help='Random erase count (default: 1)')
    parser.add_argument('--resplit', action='store_true', default=False,
                        help='Do not random erase first (clean) augmentation split')

    # * Mixup params
    parser.add_argument('--mixup', type=float, default=0.8, # JY
                        help='mixup alpha, mixup enabled if > 0.')
    parser.add_argument('--cutmix', type=float, default=1.0, # JY
                        help='cutmix alpha, cutmix enabled if > 0.')
    parser.add_argument('--cutmix_minmax', type=float, nargs='+', default=None,
                        help='cutmix min/max ratio, overrides alpha and enables cutmix if set (default: None)')
    parser.add_argument('--mixup_prob', type=float, default=1.0,
                        help='Probability of performing mixup or cutmix when either/both is enabled')
    parser.add_argument('--mixup_switch_prob', type=float, default=0.5,
                        help='Probability of switching to cutmix when both mixup and cutmix enabled')
    parser.add_argument('--mixup_mode', type=str, default='batch',
                        help='How to apply mixup/cutmix params. Per "batch", "pair", or "elem"')

    # * Finetuning params
    parser.add_argument('--finetune', default='', # JY
                        help='finetune from checkpoint')
    parser.add_argument('--global_pool', action='store_true')
    parser.set_defaults(global_pool=True)
    parser.add_argument('--cls_token', action='store_false', dest='global_pool',
                        help='Use class token instead of global pool for classification')

    # Dataset parameters
    parser.add_argument('--data_path', default='/data4/jiangy/images_classic/cifar10/', type=str,
                        help='dataset path')
    parser.add_argument('--nb_classes', default=10, type=int,
                        help='number of the classification types')

    parser.add_argument('--output_dir', default='./output_dir/test_parallel',
                        help='path where to save, empty for no saving')
    parser.add_argument('--log_dir', default='./output_dir/test_parallel',
                        help='path where to tensorboard log')
    
    parser.add_argument('--device', default='cuda',
                        help='device to use for training / testing')
    parser.add_argument('--seed', default=0, type=int)
    parser.add_argument('--resume', default='',
                        help='resume from checkpoint')

    parser.add_argument('--start_epoch', default=0, type=int, metavar='N',
                        help='start epoch')
    parser.add_argument('--eval', action='store_true',
                        help='Perform evaluation only')
    parser.add_argument('--dist_eval', action='store_true', default=False,
                        help='Enabling distributed evaluation (recommended during training for faster monitor')
    parser.add_argument('--num_workers', default=8, type=int)
    parser.add_argument('--pin_mem', action='store_true',
                        help='Pin CPU memory in DataLoader for more efficient (sometimes) transfer to GPU.')
    parser.add_argument('--no_pin_mem', action='store_false', dest='pin_mem')
    parser.set_defaults(pin_mem=True)

    # distributed training parameters
    parser.add_argument('--world_size', default=1, type=int,
                        help='number of distributed processes')
    parser.add_argument('--local_rank', default=-1, type=int)
    parser.add_argument('--dist_on_itp', action='store_true')
    parser.add_argument('--dist_url', default='env://',
                        help='url used to set up distributed training')
    
    # new parameters
    parser.add_argument('--ds_type', default='self-supervised', type=str)
    parser.add_argument('--ss', default=True, action='store_false', help='enable self-supervised training simultaneously')
    parser.add_argument('--no_aug', default=False, type=bool)
    parser.add_argument('--ss_ratio', default=1.0, type=float, help='ratio of imagenet-1k')
    parser.add_argument('--is_gray', default=False, type=bool)
    parser.add_argument('--ssl_dataset', default='imagenet-1k', type=str, help='dataset for self-supervised learning')
    parser.add_argument('--longtail_idx', default=-1, type=int, help='choose class for long-tail experiment')
    parser.add_argument('--seq', default=False, action='store_true')
    parser.add_argument('--ft_mask', action='store_true', default=False)
    parser.add_argument('--save_freq', default=5, type=int, help='save model frequency')

    return parser


def main(args):
    misc.init_distributed_mode(args)

    print('job dir: {}'.format(os.path.dirname(os.path.realpath(__file__))))
    print("{}".format(args).replace(', ', ',\n'))

    device = torch.device(args.device)
    if not args.distributed:
        torch.cuda.set_device(device)

    # fix the seed for reproducibility
    seed = args.seed + misc.get_rank()
    torch.manual_seed(seed)
    np.random.seed(seed)

    cudnn.benchmark = True

    # if ('mnist' in args.data_path):
    #     dataset_train = CustomDataset_ImgList(root='/data3/jiangy/project/openood/OpenOOD-main/data/images_classic', file_path='/data3/jiangy/project/openood/OpenOOD-main/data/benchmark_imglist/{}/train_{}.txt'.format('mnist', 'mnist'), transform=build_transform(True, args, False))
    #     dataset_val = CustomDataset_ImgList(root='/data3/jiangy/project/openood/OpenOOD-main/data/images_classic', file_path='/data3/jiangy/project/openood/OpenOOD-main/data/benchmark_imglist/{}/test_{}.txt'.format('mnist', 'mnist'), transform=build_transform(False, args, False))
    # elif ('imagenet200' in args.data_path):
    #     dataset_train = CustomDataset_ImgList(root='/data4/jiangy/OpenOOD-main/data/images_largescale', file_path='/data3/jiangy/project/openood/OpenOOD-main/data/benchmark_imglist/{}/train_{}.txt'.format('imagenet200', 'imagenet200'), transform=build_transform(True, args, False))
    #     dataset_val = CustomDataset_ImgList(root='/data4/jiangy/OpenOOD-main/data/images_largescale', file_path='/data3/jiangy/project/openood/OpenOOD-main/data/benchmark_imglist/{}/test_{}.txt'.format('imagenet200', 'imagenet200'), transform=build_transform(False, args, False))
    t = None
    for s in ['mnist', 'imagenet200', 'covid']:
        if s in args.data_path:
            t = s
    if t:
        def get_root(data):
            if data == 'mnist':
                return '/data4/jiangy/OpenOOD-main/data/images_classic'
            elif data == 'imagenet200':
                return '/data4/jiangy/OpenOOD-main/data/images_largescale'
            else:
                return '/data4/jiangy/OpenOOD-main/data/images_medical'
        dataset_train = CustomDataset_ImgList(root=get_root(t), file_path='/data4/jiangy/OpenOOD-main/data/benchmark_imglist/{}/train_{}.txt'.format(t, t if t != 'covid' else 'bimcv'), transform=build_transform(True, args, False))
        dataset_val = CustomDataset_ImgList(root=get_root(t), file_path='/data4/jiangy/OpenOOD-main/data/benchmark_imglist/{}/test_{}.txt'.format(t, t if t != 'covid' else 'bimcv'), transform=build_transform(False, args, False))
    else:
        dataset_train = build_dataset(is_train=True, args=args)
        if args.longtail_idx != -1:
            dataset_train = util.datasets.CustomDataset_longtail(os.path.join(args.data_path, 'train'), util.datasets.build_transform(is_train=True, args=args), sample_dict_path='cifar10_sample_dict_all_ratio0.2.npy', reserve_idx=args.longtail_idx)
        if args.no_aug:
            dataset_train = datasets.ImageFolder(os.path.join(args.data_path, 'train'), transform=util.datasets.build_transform(is_train=False, args=args))
        dataset_val = build_dataset(is_train=False, args=args)
    
    # transform_ss = transforms.Compose([
    #         transforms.RandomResizedCrop(args.input_size, scale=(0.2, 1.0), interpolation=3),  # 3 is bicubic
    #         transforms.RandomHorizontalFlip(),
    #         transforms.ToTensor(),
    #         transforms.Normalize(mean=[0.485, 0.456, 0.406], std=[0.229, 0.224, 0.225])])
    # dataset_train = datasets.ImageFolder(os.path.join(args.data_path, 'train'), transform=transform_ss)
    # dataset_val = datasets.ImageFolder(os.path.join(args.data_path, 'test'), transform=transform_ss)
    # dataset_ss = datasets.ImageFolder(os.path.join('/data3/jiangy/project/openood/data/imagenet', 'train'), transform=transform_ss)
    if args.ssl_dataset == 'imagenet-1k':
        dataset_ss = build_dataset(is_train=False, args=args, data_path=os.path.join('/data4/jiangy/data/imagenet', 'train')) #'/data3/jiangy/project/openood/data/imagenet', 'train'))
        # dataset_ss = build_dataset(is_train=False, args=args, data_path=os.path.join('/data4/jiangy/data/imagenet', 'train'))
    elif args.ssl_dataset == 'id':
        dataset_ss = build_dataset(is_train=False, args=args, data_path=os.path.join(args.data_path, 'train'))
    # elif args.ssl_dataset == 'cifar100':
    else:
        dataset_ss = build_dataset(is_train=False, args=args, data_path='/data4/jiangy/images_classic/{}'.format(args.ssl_dataset))
    # print(dataset_train, dataset_val, dataset_ss)

    if True:  # args.distributed:
        num_tasks = misc.get_world_size()
        global_rank = misc.get_rank()
        sampler_train = torch.utils.data.DistributedSampler(
            dataset_train, num_replicas=num_tasks, rank=global_rank, shuffle=True
        )
        print("Sampler_train = %s" % str(sampler_train))
        sampler_ss = torch.utils.data.DistributedSampler(
            dataset_ss, num_replicas=num_tasks, rank=global_rank, shuffle=True
        )
        print("Sampler_ss = %s" % str(sampler_ss))
        if args.dist_eval:
            if len(dataset_val) % num_tasks != 0:
                print('Warning: Enabling distributed evaluation with an eval dataset not divisible by process number. '
                      'This will slightly alter validation results as extra duplicate entries are added to achieve '
                      'equal num of samples per-process.')
            sampler_val = torch.utils.data.DistributedSampler(
                dataset_val, num_replicas=num_tasks, rank=global_rank, shuffle=True)  # shuffle=True to reduce monitor bias
        else:
            sampler_val = torch.utils.data.SequentialSampler(dataset_val)
            # sampler_ss = torch.utils.data.RandomSampler(dataset_ss)
    else:
        sampler_train = torch.utils.data.RandomSampler(dataset_train)
        sampler_val = torch.utils.data.SequentialSampler(dataset_val)
        sampler_ss = torch.utils.data.RandomSampler(dataset_ss)

    if global_rank == 0 and args.log_dir is not None and not args.eval:
        os.makedirs(args.log_dir, exist_ok=True)
        log_writer = SummaryWriter(log_dir=args.log_dir)
    else:
        log_writer = None

    data_loader_train = torch.utils.data.DataLoader(
        dataset_train, sampler=sampler_train,
        batch_size=args.batch_size,
        num_workers=args.num_workers,
        pin_memory=args.pin_mem,
        drop_last=True,
    )

    data_loader_val = torch.utils.data.DataLoader(
        dataset_val, sampler=sampler_val,
        batch_size=args.batch_size,
        num_workers=args.num_workers,
        pin_memory=args.pin_mem,
        drop_last=False
    )

    data_loader_ss = torch.utils.data.DataLoader(
        dataset_ss, sampler=sampler_ss,
        batch_size=int(args.batch_size * args.ss_ratio),
        num_workers=args.num_workers,
        pin_memory=args.pin_mem,
        drop_last=True
    )

    mixup_fn = None
    mixup_active = args.mixup > 0 or args.cutmix > 0. or args.cutmix_minmax is not None
    if mixup_active:
        print("Mixup is activated!")
        mixup_fn = Mixup(
            mixup_alpha=args.mixup, cutmix_alpha=args.cutmix, cutmix_minmax=args.cutmix_minmax,
            prob=args.mixup_prob, switch_prob=args.mixup_switch_prob, mode=args.mixup_mode,
            label_smoothing=args.smoothing, num_classes=args.nb_classes)
        mixup_fn_ss = Mixup(
            mixup_alpha=args.mixup, cutmix_alpha=args.cutmix, cutmix_minmax=args.cutmix_minmax,
            prob=args.mixup_prob, switch_prob=args.mixup_switch_prob, mode=args.mixup_mode,
            label_smoothing=args.smoothing, num_classes=1000)
    print("=" * 20 , "\n", "Model: ", args.model)
    if 'mae' not in args.model:
        print("=" * 20 , "\n", "Not MAE. Model: ", args.model)
        model = models_vit.__dict__[args.model](
            num_classes=args.nb_classes,
            drop_path_rate=args.drop_path,
            global_pool=args.global_pool,
        )
    else:
        model = models_mae.__dict__[args.model](norm_pix_loss=args.norm_pix_loss)
        model.set_ft_mask(args.ft_mask, args.mask_ratio)
    model.num_classes = args.nb_classes
    model.head = torch.nn.Linear(model.embed_dim, model.num_classes)
    # model.head = torch.nn.Linear(model.embed_dim, model.num_classes).to(device) #torch.nn.Sequential(torch.nn.BatchNorm1d(model.embed_dim, affine=False, eps=1e-6), torch.nn.Linear(model.embed_dim, model.num_classes)).to(device)
    print("Num Classes: ", args.nb_classes)
    # model.head_in1k = torch.nn.Linear(model.embed_dim, 1000).to(device) 
    

    if args.finetune and not args.eval:
        checkpoint = torch.load(args.finetune, map_location='cpu')

        print("Load pre-trained checkpoint from: %s" % args.finetune)
        checkpoint_model = checkpoint['model']
        state_dict = model.state_dict()
        for k in ['head.weight', 'head.bias']:
            if k in checkpoint_model and checkpoint_model[k].shape != state_dict[k].shape:
                print(f"Removing key {k} from pretrained checkpoint")
                del checkpoint_model[k]

        # interpolate position embedding
        interpolate_pos_embed(model, checkpoint_model)

        # load pre-trained model
        msg = model.load_state_dict(checkpoint_model, strict=False)
        print(msg)

        if args.global_pool:
            # assert set(msg.missing_keys) == {'head.weight', 'head.bias', 'fc_norm.weight', 'fc_norm.bias'}
            pass
            # assert set(msg.missing_keys) == {'head.weight', 'head.bias', 'fc_norm.weight', 'fc_norm.bias', 'head_in1k.weight', 'head_in1k.bias'}
        else:
            assert set(msg.missing_keys) == {'head.weight', 'head.bias'}

        # manually initialize fc layer
        trunc_normal_(model.head.weight, std=2e-5)

    model.to(device)

    model_without_ddp = model
    n_parameters = sum(p.numel() for p in model.parameters() if p.requires_grad)

    # print("Model = %s" % str(model_without_ddp))
    print('number of params (M): %.2f' % (n_parameters / 1.e6))

    eff_batch_size = args.batch_size * args.accum_iter * misc.get_world_size()
    
    if args.lr is None:  # only base_lr is specified
        args.lr = args.blr * eff_batch_size / 256

    print("base lr: %.2e" % (args.lr * 256 / eff_batch_size))
    print("actual lr: %.2e" % args.lr)

    print("accumulate grad iterations: %d" % args.accum_iter)
    print("effective batch size: %d" % eff_batch_size)

    if args.distributed:
        model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.gpu])
        model_without_ddp = model.module

    # build optimizer with layer-wise lr decay (lrd)
    param_groups = lrd.param_groups_lrd(model_without_ddp, args.weight_decay,
        no_weight_decay_list=model_without_ddp.no_weight_decay(),
        layer_decay=args.layer_decay
    )
    optimizer = torch.optim.AdamW(param_groups, lr=args.lr)
    loss_scaler = NativeScaler()
    if args.seq:
        param_groups_ss = optim_factory.add_weight_decay(model_without_ddp, args.weight_decay)
        optimizer_ss = torch.optim.AdamW(param_groups_ss, lr=args.lr, betas=(0.9, 0.95))
        loss_scaler_ss = NativeScaler()

    if mixup_fn is not None:
        # smoothing is handled with mixup label transform
        criterion = SoftTargetCrossEntropy()
    elif args.smoothing > 0.:
        criterion = LabelSmoothingCrossEntropy(smoothing=args.smoothing)
    else:
        criterion = torch.nn.CrossEntropyLoss()

    print("criterion = %s" % str(criterion))

    misc.load_model(args=args, model_without_ddp=model_without_ddp, optimizer=optimizer, loss_scaler=loss_scaler)

    # beta_dist = sp.stats.beta(0.5, 0.5)
    # total_iter = 100 * len(data_loader_train)
    # weight_func = lambda it: beta_dist.pdf((it + 0.5) / (total_iter + 1))

    bma_model = None # GeneralMovingAverage(model_without_ddp, weight_func)

    if args.eval:
        test_stats = evaluate_new(data_loader_val, model, device)
        print(f"Accuracy of the network on the {len(dataset_val)} test images: {test_stats['acc1']:.1f}%")
        exit(0)

    print("USE self-supervised learning" if args.ss else "No self-supervised")

    print(f"Start training for {args.epochs} epochs")
    start_time = time.time()
    max_accuracy = 0.0
    for epoch in range(args.start_epoch, args.epochs):
        if args.distributed:
            data_loader_train.sampler.set_epoch(epoch)
        # train_stats = train_one_epoch_ss(
        #     model, criterion, data_loader_train, data_loader_ss,
        #     optimizer, device, epoch, loss_scaler,
        #     args.clip_grad, mixup_fn, mixup_fn_ss, bma_model,
        #     log_writer=log_writer,
        #     args=args
        # )
        if 'mae' not in args.model:
            train_stats = train_one_epoch(
                model, criterion, data_loader_train,
                optimizer, device, epoch, loss_scaler,
                args.clip_grad, mixup_fn,
                log_writer=log_writer,
                args=args
            )
        elif args.seq:
            train_stats = train_one_epoch_ss_seq_parallel(
                model, criterion, data_loader_train, data_loader_ss,
                optimizer, device, epoch, loss_scaler, loss_scaler_ss,
                args.clip_grad, mixup_fn, mixup_fn_ss, bma_model,
                log_writer=log_writer,
                args=args
            )
        else:
            train_stats = train_one_epoch_ss_parallel(
                model, criterion, data_loader_train, data_loader_ss,
                optimizer, device, epoch, loss_scaler,
                args.clip_grad, mixup_fn, mixup_fn_ss, bma_model,
                log_writer=log_writer,
                args=args
            )
        # if args.ss:
        #     train_stats_ss = engine_pretrain.train_one_epoch(
        #         model, data_loader_ss,
        #         optimizer, device, epoch, loss_scaler,
        #         log_writer=log_writer,
        #         args=args
        #     )
        if (epoch == 9 or (epoch + 1) % args.save_freq == 0 or (args.nb_classes == 2 and epoch < 30) or args.nb_classes == 1000) and args.output_dir:
            misc.save_model(
                args=args, model=model, model_without_ddp=model_without_ddp, optimizer=optimizer,
                loss_scaler=loss_scaler, epoch=epoch)
            # torch.save(bma_model.state_dict(), os.path.join(args.output_dir, 'bma.pth'))
            # if epoch == 99:
            #     torch.save(bma_model.state_dict(), os.path.join(args.output_dir, 'bma-99.pth'))

        test_stats = evaluate_new(data_loader_val, model, device, args)
        print(f"Accuracy of the network on the {len(dataset_val)} test images: {test_stats['acc1']:.1f}%")
        max_accuracy = max(max_accuracy, test_stats["acc1"])
        print(f'Max accuracy: {max_accuracy:.2f}%')

        if log_writer is not None:
            log_writer.add_scalar('perf/test_acc1', test_stats['acc1'], epoch)
            log_writer.add_scalar('perf/test_acc5', test_stats['acc5'], epoch)
            log_writer.add_scalar('perf/test_loss', test_stats['loss'], epoch)

        log_stats = {**{f'train_{k}': v for k, v in train_stats.items()},
                        **{f'test_{k}': v for k, v in test_stats.items()},
                        'epoch': epoch,
                        'n_parameters': n_parameters}
        
        # if args.ss:
        #     log_stats = {**{f'train_{k}': v for k, v in train_stats.items()},
        #                 **{f'train_ss_{k}': v for k, v in train_stats_ss.items()},
        #                     **{f'test_{k}': v for k, v in test_stats.items()},
        #                     'epoch': epoch,
        #                     'n_parameters': n_parameters}


        if args.output_dir and misc.is_main_process():
            if log_writer is not None:
                log_writer.flush()
            with open(os.path.join(args.output_dir, "log.txt"), mode="a", encoding="utf-8") as f:
                f.write(json.dumps(log_stats) + "\n")

    total_time = time.time() - start_time
    total_time_str = str(datetime.timedelta(seconds=int(total_time)))
    print('Training time {}'.format(total_time_str))


if __name__ == '__main__':
    args = get_args_parser()
    args = args.parse_args()
    if args.output_dir:
        Path(args.output_dir).mkdir(parents=True, exist_ok=True)
    main(args)
