"""
Misc functions, including distributed helpers.

Mostly copy-paste from torchvision references.
"""
import io
import os
import time
from collections import defaultdict, deque, OrderedDict
import datetime

import torch
import torch.distributed as dist

def create_optimizer(args, model):

    if args.transform_model:
        gating_params, other_params = split_dict(
            OrderedDict(model.named_parameters()),
            lambda name: ('alpha' in name)# or ('pos' in name)
        )
        parameters =  [
                    {"params": filter(lambda p : p.requires_grad, gating_params.values()), "lr":args.gating_lr, "weight_decay":0},
                    {"params": filter(lambda p : p.requires_grad, other_params.values())}
                ]
    else:
        parameters = filter(lambda p : p.requires_grad, model.parameters())

    if args.opt == "SGD":
        optimizer = torch.optim.SGD(parameters, lr=args.lr, momentum=args.momentum, weight_decay=args.weight_decay)
    else:
        optimizer = getattr(torch.optim, args.opt)(parameters, lr=args.lr, weight_decay=args.weight_decay)
    return optimizer



def get_rel_indices(num_patches, locality_dim=1):
    img_size = int(num_patches**.5)
    rel_indices   = torch.zeros(1, num_patches, num_patches, 3*locality_dim)
    ind = torch.arange(img_size).view(1,-1) - torch.arange(img_size).view(-1, 1)
    indx = ind.repeat(img_size,img_size)
    indy = ind.repeat_interleave(img_size,dim=0).repeat_interleave(img_size,dim=1)
    indd = indx**2 + indy**2
    rel_indices[:,:,:,:3*locality_dim] = indd.unsqueeze(-1)
    rel_indices[:,:,:,:2*locality_dim] = indy.unsqueeze(-1)
    rel_indices[:,:,:,:1*locality_dim] = indx.unsqueeze(-1)
    return rel_indices

def split_dict(d, first_predicate):
    """split the dictionary d into 2 dictionaries, first one contains elements validating first_predicate"""
    first, second = OrderedDict(), OrderedDict()
    for key, value in d.items():
        if first_predicate(key):
            first[key] = value
        else:
            second[key] = value
    return first, second

class SmoothedValue(object):
    """Track a series of values and provide access to smoothed values over a
    window or the global series average.
    """

    def __init__(self, window_size=20, fmt=None):
        if fmt is None:
            fmt = "{median:.4f} ({global_avg:.4f})"
        self.deque = deque(maxlen=window_size)
        self.total = 0.0
        self.count = 0
        self.fmt = fmt

    def update(self, value, n=1):
        self.deque.append(value)
        self.count += n
        self.total += value * n

    def synchronize_between_processes(self):
        """
        Warning: does not synchronize the deque!
        """
        if not is_dist_avail_and_initialized():
            return
        t = torch.tensor([self.count, self.total], dtype=torch.float64, device='cuda')
        dist.barrier()
        dist.all_reduce(t)
        t = t.tolist()
        self.count = int(t[0])
        self.total = t[1]

    @property
    def median(self):
        d = torch.tensor(list(self.deque))
        return d.median().item()

    @property
    def avg(self):
        d = torch.tensor(list(self.deque), dtype=torch.float32)
        return d.mean().item()

    @property
    def global_avg(self):
        return self.total / self.count

    @property
    def max(self):
        return max(self.deque)

    @property
    def value(self):
        return self.deque[-1]

    def __str__(self):
        return self.fmt.format(
            median=self.median,
            avg=self.avg,
            global_avg=self.global_avg,
            max=self.max,
            value=self.value)


class MetricLogger(object):
    def __init__(self, delimiter="\t"):
        self.meters = defaultdict(SmoothedValue)
        self.delimiter = delimiter

    def update(self, **kwargs):
        for k, v in kwargs.items():
            if isinstance(v, torch.Tensor):
                v = v.item()
            assert isinstance(v, (float, int))
            self.meters[k].update(v)

    def __getattr__(self, attr):
        if attr in self.meters:
            return self.meters[attr]
        if attr in self.__dict__:
            return self.__dict__[attr]
        raise AttributeError("'{}' object has no attribute '{}'".format(
            type(self).__name__, attr))

    def __str__(self):
        loss_str = []
        for name, meter in self.meters.items():
            loss_str.append(
                "{}: {}".format(name, str(meter))
            )
        return self.delimiter.join(loss_str)

    def synchronize_between_processes(self):
        for meter in self.meters.values():
            meter.synchronize_between_processes()

    def add_meter(self, name, meter):
        self.meters[name] = meter

    def log_every(self, iterable, print_freq, header=None):
        i = 0
        if not header:
            header = ''
        start_time = time.time()
        end = time.time()
        iter_time = SmoothedValue(fmt='{avg:.4f}')
        data_time = SmoothedValue(fmt='{avg:.4f}')
        space_fmt = ':' + str(len(str(len(iterable)))) + 'd'
        log_msg = [
            header,
            '[{0' + space_fmt + '}/{1}]',
            'eta: {eta}',
            '{meters}',
            'time: {time}',
            'data: {data}'
        ]
        if torch.cuda.is_available():
            log_msg.append('max mem: {memory:.0f}')
        log_msg = self.delimiter.join(log_msg)
        MB = 1024.0 * 1024.0
        for obj in iterable:
            data_time.update(time.time() - end)
            yield obj
            iter_time.update(time.time() - end)
            if i % print_freq == 0 or i == len(iterable) - 1:
                eta_seconds = iter_time.global_avg * (len(iterable) - i)
                eta_string = str(datetime.timedelta(seconds=int(eta_seconds)))
                if torch.cuda.is_available():
                    print(log_msg.format(
                        i, len(iterable), eta=eta_string,
                        meters=str(self),
                        time=str(iter_time), data=str(data_time),
                        memory=torch.cuda.max_memory_allocated() / MB))
                else:
                    print(log_msg.format(
                        i, len(iterable), eta=eta_string,
                        meters=str(self),
                        time=str(iter_time), data=str(data_time)))
            i += 1
            end = time.time()
        total_time = time.time() - start_time
        total_time_str = str(datetime.timedelta(seconds=int(total_time)))
        print('{} Total time: {} ({:.4f} s / it)'.format(
            header, total_time_str, total_time / len(iterable)))


def _load_checkpoint_for_ema(model_ema, checkpoint):
    """
    Workaround for ModelEma._load_checkpoint to accept an already-loaded object
    """
    mem_file = io.BytesIO()
    torch.save(checkpoint, mem_file)
    mem_file.seek(0)
    model_ema._load_checkpoint(mem_file)


def setup_for_distributed(is_master):
    """
    This function disables printing when not in master process
    """
    import builtins as __builtin__
    builtin_print = __builtin__.print

    def print(*args, **kwargs):
        force = kwargs.pop('force', False)
        if is_master or force:
            builtin_print(*args, **kwargs)

    __builtin__.print = print


def is_dist_avail_and_initialized():
    if not dist.is_available():
        return False
    if not dist.is_initialized():
        return False
    return True


def get_world_size():
    if not is_dist_avail_and_initialized():
        return 1
    return dist.get_world_size()


def get_rank():
    if not is_dist_avail_and_initialized():
        return 0
    return dist.get_rank()


def is_main_process():
    return get_rank() == 0


def save_on_master(*args, **kwargs):
    if is_main_process():
        torch.save(*args, **kwargs)


def init_distributed_mode(args):
    if 'RANK' in os.environ and 'WORLD_SIZE' in os.environ:
        args.rank = int(os.environ["RANK"])
        args.world_size = int(os.environ['WORLD_SIZE'])
        args.gpu = int(os.environ['LOCAL_RANK'])
    elif 'SLURM_PROCID' in os.environ:
        args.rank = int(os.environ['SLURM_PROCID'])
        args.gpu = args.rank % torch.cuda.device_count()
    else:
        print('Not using distributed mode')
        args.distributed = False
        return

    args.distributed = True

    torch.cuda.set_device(args.gpu)
    args.dist_backend = 'nccl'
    print('| distributed init (rank {}): {}'.format(
        args.rank, args.dist_url), flush=True)
    torch.distributed.init_process_group(backend=args.dist_backend, init_method=args.dist_url,
                                         world_size=args.world_size, rank=args.rank)
    torch.distributed.barrier()
    setup_for_distributed(args.rank == 0)

def compute_throughput(model, batch_size=128, resolution=224, train=False, optimizer=None):

    if not train:
        torch.no_grad()
    torch.cuda.empty_cache()
    warmup_iters = 3
    num_iters = 10
    if train:
        model.train()
        crit = torch.nn.CrossEntropyLoss()
        def func():
            out = model(inputs)
            loss = crit(out, targets)
            loss.backward()
            # optimizer.step()
    else:
        model.eval()
        def func():
            model(inputs)
    model.to("cuda")
    timing = []    

    inputs = torch.randn(batch_size, 3, resolution, resolution, device="cuda")
    targets = torch.ones(batch_size, device="cuda").long()

    # warmup
    for _ in range(warmup_iters):
        func()

    torch.cuda.synchronize()
    for _ in range(num_iters):
        start = time.time()
        func()
        torch.cuda.synchronize()
        timing.append(time.time() - start)

    timing = torch.as_tensor(timing, dtype=torch.float32)
    return batch_size / timing.mean()

@torch.no_grad()
def profile_model(model, batch_size=128, resolution=224):

    model.to("cuda")
    inputs = torch.randn(batch_size, 3, resolution, resolution, device="cuda")
    with torch.autograd.profiler.profile(use_cuda=True) as prof:
        model(inputs)
    return  prof.key_averages().table(sort_by="cpu_time_total", row_limit=10)

def get_grad_norm(model):

    total_norm = 0
    for p in model.parameters():
        param_norm = p.grad.data.norm(2)
        total_norm += param_norm.item() ** 2
    return total_norm**0.5
