import torch
import torch.nn as nn

from set_model import MeanPooling
from algorithms import lr
from models.get_model import get_model
from utils import adjust_learning_rate

def off_diagonal(x):
    # return a flattened view of the off-diagonal elements of a square matrix
    n, m = x.shape
    assert n == m
    return x.flatten()[:-1].view(n - 1, n + 1)[:, 1:].flatten()

def return_loss_fn(x, y, t=0.0051):
    N, D = x.shape
    c = x.T @ y
    c.div_(N)
    on_diag = torch.diagonal(c).add_(-1).pow_(2).sum()
    off_diag = off_diagonal(c).pow_(2).sum()
    loss = on_diag + t*off_diag    
    return loss

def run(args, train_loader, meta_val_ds, meta_test_ds, device, logger):
    # data    
    transform = args.transform

    # model
    encoder = get_model(args.model, args.img_size).to(device)
    last_hidden_size = encoder(
        torch.randn(1, 3, args.img_size, args.img_size).to(device)).shape[-1]    
    decoder = MeanPooling()

    head = nn.Sequential(
        nn.Linear(last_hidden_size, last_hidden_size),
        nn.BatchNorm1d(last_hidden_size),
        nn.LeakyReLU(),
        nn.Linear(last_hidden_size, int(last_hidden_size/4))
    ).to(device)

    bn = nn.BatchNorm1d(int(last_hidden_size/4), affine=False).to(device)

    logger.register_model_to_save(encoder, 'encoder')
    logger.register_model_to_save(decoder, 'decoder')
    logger.register_model_to_save(head, 'head')
    logger.register_model_to_save(bn, 'bn')
    
    optimizer = torch.optim.Adam(
        [{"params": encoder.parameters(), "lr": args.lr},
        {"params": decoder.parameters(), "lr": args.lr},
        {"params": head.parameters(), "lr": args.lr},
        {"params": bn.parameters(), "lr": args.lr}]
    )

    step = 1
    for epoch in range(1, args.training_epochs+1):
        for data in train_loader:
            encoder.train(), decoder.train(), head.train()
            optimizer.zero_grad()

            # lr scheduling
            if args.lr_scheduling:
                adjust_learning_rate(args, optimizer, train_loader, step)
            
            # data augmentation
            data = data.to(device)
            data_augmented = torch.stack([data]*2, dim=1)            
            data_augmented = data_augmented.reshape(-1, 3, args.pre_img_size, args.pre_img_size)
            data_augmented = transform(data_augmented)
            data = data_augmented.reshape(-1, 2, 3, args.img_size, args.img_size)   
            d = data.size()

            # instance feature            
            features = encoder(
                data.reshape(d[0]*2, d[2], d[3], d[4]).contiguous()
            )
            embeddings = bn(head(
                features
            )).reshape(d[0], 2, -1).contiguous()
            
            # ins vs ins loss
            loss = return_loss_fn(
                embeddings[:, 0, :].contiguous(),
                embeddings[:, 1, :].contiguous()
            )
            loss.backward()
            optimizer.step()

            logger.meter('train', 'ins vs ins loss', loss)
            step += 1

            if args.debug:
                break
        
        if epoch % args.test_every == 0:
            args.support = 5
            mean, ci = lr.run(args, args.val_episodes, encoder, decoder, meta_val_ds, device)        
            logger.meter('5shot-val', 'accuracy', mean)
            logger.meter('5shot-val', 'ci', ci)
        
        if epoch == args.training_epochs:
            for support in [1, 5, 20, 50]:
                args.support = support
                mean, ci = lr.run(args, args.test_episodes, encoder, decoder, meta_test_ds, device)
                logger.meter(f'{support}shot-test', 'accuracy', mean)
                logger.meter(f'{support}shot-test', 'ci', ci)

        logger.step()