import torch
import torch.nn as nn

from torch.utils.data import TensorDataset, DataLoader
from set_model import MeanPooling
from algorithms import lr
from models.get_model import get_model
from utils import adjust_learning_rate

def ema(model, model_ema, t=0.99):
    for w, fw in zip(model.parameters(), model_ema.parameters()):
        fw.data.mul_(t).add_(w.data, alpha=1.-t)

def return_loss_fn(x, y):
    y = y.detach()
    x_norm = torch.norm(x, p=2, dim=1)
    y_norm = torch.norm(y, p=2, dim=1)
    mse = 2. - 2.*torch.sum(x*y, dim=1)/(x_norm*y_norm)
    return mse.mean()

def run(args, train_loader, meta_val_ds, meta_test_ds, device, logger):
    # data    
    transform = args.transform

    # model
    encoder = get_model(args.model, args.img_size).to(device)
    last_hidden_size = encoder(
        torch.randn(1, 3, args.img_size, args.img_size).to(device)).shape[-1]

    encoder_ema = get_model(args.model, args.img_size).to(device)
    encoder_ema.load_state_dict(encoder.state_dict(), strict=True)
    
    decoder = MeanPooling()    

    head = nn.Sequential(
        nn.Linear(last_hidden_size, last_hidden_size),
        nn.BatchNorm1d(last_hidden_size),
        nn.LeakyReLU(),
        nn.Linear(last_hidden_size, int(last_hidden_size/4)),
    ).to(device)

    head_ema = nn.Sequential(
        nn.Linear(last_hidden_size, last_hidden_size),
        nn.BatchNorm1d(last_hidden_size),
        nn.LeakyReLU(),
        nn.Linear(last_hidden_size, int(last_hidden_size/4)),
    ).to(device)
    head_ema.load_state_dict(head.state_dict(), strict=True)

    predictor = nn.Sequential(
        nn.Linear(int(last_hidden_size/4), int(last_hidden_size/4)),
        nn.BatchNorm1d(int(last_hidden_size/4)),
        nn.LeakyReLU(),
        nn.Linear(int(last_hidden_size/4), int(last_hidden_size/4)),
    ).to(device)

    logger.register_model_to_save(encoder, 'encoder')
    logger.register_model_to_save(decoder, 'decoder')    
    logger.register_model_to_save(head, 'head')
    logger.register_model_to_save(predictor, 'predictor')
    
    optimizer = torch.optim.Adam(
        [{"params": encoder.parameters(), "lr": args.lr},
        {"params": decoder.parameters(), "lr": args.lr},        
        {"params": head.parameters(), "lr": args.lr},
        {"params": predictor.parameters(), "lr": args.lr}]
    )

    step = 1
    for epoch in range(1, args.training_epochs+1):
        for data in train_loader:
            encoder.train(), decoder.train(), head.train()
            optimizer.zero_grad()

            # lr scheduling
            if args.lr_scheduling:
                adjust_learning_rate(args, optimizer, train_loader, step)
            
            # data augmentation
            data = data.to(device)
            data_augmented = torch.stack([data]*2, dim=1)            
            data_augmented = data_augmented.reshape(-1, 3, args.pre_img_size, args.pre_img_size)
            data_augmented = transform(data_augmented)
            data = data_augmented.reshape(-1, 2, 3, args.img_size, args.img_size)   
            d = data.size()

            # ins vs ins loss           
            features = encoder(
                data.reshape(d[0]*2, d[2], d[3], d[4]).contiguous()
            )
            embeddings = predictor(head(
                features
            )).reshape(d[0], 2, -1).contiguous()

            # ema feature
            with torch.no_grad():
                features_ema = encoder_ema(
                    data.reshape(d[0]*2, d[2], d[3], d[4]).contiguous()
                )
                embeddings_ema = head_ema(
                    features_ema
                ).reshape(d[0], 2, -1).contiguous()

            # ins vs ins loss
            loss = return_loss_fn(
                embeddings[:, 0, :].contiguous(),
                embeddings_ema[:, 1, :].contiguous()
            )
            loss += return_loss_fn(
                embeddings[:, 1, :].contiguous(),
                embeddings_ema[:, 0, :].contiguous()
            )
            loss /= 2.
            loss.backward()
            optimizer.step()

            # ema update
            ema(encoder, encoder_ema)
            ema(head, head_ema)

            logger.meter('train', 'ins vs ins loss', loss)
            step += 1

            if args.debug:
                break
            
        if epoch % args.test_every == 0:
            args.support = 5
            mean, ci = lr.run(args, args.val_episodes, encoder, decoder, meta_val_ds, device)        
            logger.meter('5shot-val', 'accuracy', mean)
            logger.meter('5shot-val', 'ci', ci)
        
        if epoch == args.training_epochs:
            for support in [1, 5, 20, 50]:
                args.support = support
                mean, ci = lr.run(args, args.test_episodes, encoder, decoder, meta_test_ds, device)
                logger.meter(f'{support}shot-test', 'accuracy', mean)
                logger.meter(f'{support}shot-test', 'ci', ci)

        logger.step()
            
