import tkinter as tk
import sys
import json
import os
import ttkbootstrap as tb
from tkinter import ttk, messagebox
from PIL import Image, ImageTk
sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))
from tools.lvm_pool import gpt4o

def load_data(file_path):
    if os.path.exists(file_path):
        with open(file_path, 'r') as file:
            return json.load(file)
    return []

def save_data(data, file_path):
    with open(file_path, 'w') as file:
        json.dump(data, file, indent=4)

class AlignApp:
    def __init__(self, root):
        self.root = root
        self.root.title("Human Evaluation Platform")

        self.main_frame = tb.Frame(self.root)
        self.main_frame.pack(fill=tk.BOTH, expand=1)

        self.canvas = tk.Canvas(self.main_frame)
        self.scrollbar = ttk.Scrollbar(self.main_frame, orient=tk.VERTICAL, command=self.canvas.yview)
        self.scrollable_frame = tb.Frame(self.canvas)

        self.scrollable_frame.bind(
            "<Configure>",
            lambda e: self.canvas.configure(
                scrollregion=self.canvas.bbox("all")
            )
        )

        self.canvas.create_window((0, 0), window=self.scrollable_frame, anchor="nw")
        self.canvas.configure(yscrollcommand=self.scrollbar.set)

        self.canvas.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        self.scrollbar.pack(side=tk.RIGHT, fill=tk.Y)


        self.data = {"images": []}
        self.current_index = 0
        self.user_input = ""
        self.difficulty = ""
        self.eval_type = ""
        self.data_file_path = ""
        self.save_file_path = ""

        self.create_widgets()
    
    

    def create_widgets(self):
        self.user_input_label = tb.Label(self.scrollable_frame, text="User Input:", bootstyle="secondary")
        self.user_input_label.pack(pady=5)
        self.user_input_entry = tb.Combobox(self.scrollable_frame, values=[
            "atmospheric_understanding", "basic_understanding", "reasoning_capacity",
            "semantic_understanding", "spatial_understanding"
        ], bootstyle="info")
        self.user_input_entry.pack(pady=5)

        self.difficulty_label = tb.Label(self.scrollable_frame, text="Select Difficulty:", bootstyle="secondary")
        self.difficulty_label.pack(pady=5)
        self.difficulty_combo = tb.Combobox(self.scrollable_frame, values=["easy", "medium", "hard"], bootstyle="info")
        self.difficulty_combo.pack(pady=5)

        self.eval_type_label = tb.Label(self.scrollable_frame, text="Select Eval Type:", bootstyle="secondary")
        self.eval_type_label.pack(pady=5)
        self.eval_type_combo = tb.Combobox(self.scrollable_frame, values=["aligned_image_json", "questions"], bootstyle="info")
        self.eval_type_combo.pack(pady=5)

        self.load_button = tb.Button(self.scrollable_frame, text="Load Data", command=self.load_data_file, bootstyle="primary-outline")
        self.load_button.pack(pady=10)

        self.image_label = tb.Label(self.scrollable_frame)
        self.image_label.pack(pady=10)

        self.prompt_label = tb.Label(self.scrollable_frame, text="", wraplength=400, bootstyle="info")
        self.prompt_label.pack(pady=5)

        self.prompt_translate_button = tb.Button(self.scrollable_frame, text="Translate Prompt", command=self.translate_prompt, bootstyle="info-outline")
        self.prompt_translate_button.pack(pady=5)

        self.question_label = tb.Label(self.scrollable_frame, text="", wraplength=400, bootstyle="info")
        self.question_label.pack(pady=5)

        self.question_translate_button = tb.Button(self.scrollable_frame, text="Translate Question", command=self.translate_question, bootstyle="info-outline")
        self.question_translate_button.pack(pady=5)

        self.answer_label = tb.Label(self.scrollable_frame, text="", wraplength=400, bootstyle="info")
        self.answer_label.pack(pady=5)

        self.answer_translate_button = tb.Button(self.scrollable_frame, text="Translate Answer", command=self.translate_answer, bootstyle="info-outline")
        self.answer_translate_button.pack(pady=5)

        self.status_label = tb.Label(self.scrollable_frame, text="", bootstyle="secondary")
        self.status_label.pack(pady=5)

        self.index_label = tb.Label(self.scrollable_frame, text="", bootstyle="secondary")
        self.index_label.pack(pady=5)

        self.align_button = tb.Button(self.scrollable_frame, text="Align", command=lambda: self.save_choice("align"), bootstyle="success")
        self.align_button.pack(side=tk.LEFT, padx=(20, 10), pady=10)

        self.not_align_button = tb.Button(self.scrollable_frame, text="Not Align", command=lambda: self.save_choice("not_align"), bootstyle="danger")
        self.not_align_button.pack(side=tk.LEFT, padx=(10, 20), pady=10)

        self.prev_button = tb.Button(self.scrollable_frame, text="Previous", command=self.prev_image, bootstyle="warning-outline")
        self.prev_button.pack(side=tk.LEFT, padx=(20, 10), pady=10)

        self.next_button = tb.Button(self.scrollable_frame, text="Next", command=self.next_image, bootstyle="warning-outline")
        self.next_button.pack(side=tk.LEFT, padx=(10, 20), pady=10)

        self.stats_button = tb.Button(self.scrollable_frame, text="Show Stats", command=self.show_stats, bootstyle="info-outline")
        self.stats_button.pack(side=tk.LEFT, padx=(20, 10), pady=10)

    def load_data_file(self):
        self.user_input = self.user_input_entry.get()
        self.difficulty = self.difficulty_combo.get()
        self.eval_type = self.eval_type_combo.get()

        if not self.user_input or not self.difficulty or not self.eval_type:
            messagebox.showwarning("Warning", "Please fill in all fields.")
            return

        self.save_file_path = f'./document/{self.user_input}/human_eval/{self.eval_type}/{self.difficulty}_user_choices.json'
        if not os.path.exists(f'./document/{self.user_input}/human_eval/{self.eval_type}'):
            os.makedirs(f'./document/{self.user_input}/human_eval/{self.eval_type}')
        if len(load_data(self.save_file_path)) == 0:
            self.data_file_path = f'./document/{self.user_input}/{self.eval_type}/{self.difficulty}_{self.eval_type}.json'
            if self.eval_type == "aligned_image_json":
                self.data_file_path = f'./document/{self.user_input}/{self.eval_type}/{self.difficulty}_aligned_images.json'
        else:
            self.data_file_path = self.save_file_path

        self.data = load_data(self.data_file_path)
        print(f"Loaded {len(self.data)} images.")
        self.current_index = 0
        self.show_image()

    def show_image(self):
        if self.current_index < 0 or self.current_index >= len(self.data):
            return
        item = self.data[self.current_index]
        image_path = item["image_path"]

        image = Image.open(image_path)
        image = image.resize((400, 300), Image.Resampling.LANCZOS)
        photo = ImageTk.PhotoImage(image)
        self.image_label.config(image=photo)
        self.image_label.image = photo

        self.prompt_label.config(text=f"Prompt: {item['prompt']}")
        self.question_label.config(text=f"Question: {item.get('objective_question', 'N/A')}")
        self.answer_label.config(text=f"Answer: {item.get('objective_reference_answer', 'N/A')}")

        if "user_choice" in item:
            self.status_label.config(text="Status: Marked", bootstyle="success")
        else:
            self.status_label.config(text="Status: Not Marked", bootstyle="danger")

        self.index_label.config(text=f"Image {self.current_index + 1} / {len(self.data)}")

    def save_choice(self, choice):
        if self.current_index < 0 or self.current_index >= len(self.data):
            return

        item = self.data[self.current_index]
        item["user_choice"] = choice
        save_data(self.data, self.save_file_path)
        self.current_index += 1

        self.show_image()

    def next_image(self):
        self.current_index += 1
        if self.current_index >= len(self.data):
            self.current_index = 0
        self.show_image()

    def prev_image(self):
        self.current_index -= 1
        if self.current_index < 0:
            self.current_index = len(self.data) - 1
        self.show_image()

    def show_stats(self):
        total_items = len(self.data)
        if total_items == 0:
            messagebox.showinfo("Stats", "No data available.")
            return

        align_count = sum(1 for item in self.data if item.get("user_choice") == "align")
        user_choice_count = sum(1 for item in self.data if "user_choice" in item)

        align_rate = (align_count / user_choice_count) * 100 if user_choice_count > 0 else 0

        stats_message = (
            f"Total items: {total_items}\n"
            f"Items with user choices: {user_choice_count}\n"
            f"Align rate: {align_rate:.2f}%"
        )
        messagebox.showinfo("Stats", stats_message)

    def translate_text(self, text):
        return gpt4o(f"translate English to Chinese: {text}")

    def translate_prompt(self):
        if self.current_index < 0 or self.current_index >= len(self.data):
            return
        item = self.data[self.current_index]
        prompt = item['prompt']
        translated_prompt = self.translate_text(prompt)
        self.prompt_label.config(text=f"Prompt: {translated_prompt}")
        # messagebox.showinfo("Translation", "Prompt translation completed!")

    def translate_question(self):
        if self.current_index < 0 or self.current_index >= len(self.data):
            return
        item = self.data[self.current_index]
        question = item.get('objective_question', 'N/A')
        if question != 'N/A':
            translated_question = self.translate_text(question)
            self.question_label.config(text=f"Question: {translated_question}")
            # messagebox.showinfo("Translation", "Question translation completed!")

    def translate_answer(self):
        if self.current_index < 0 or self.current_index >= len(self.data):
            return
        item = self.data[self.current_index]
        answer = item.get('objective_reference_answer', 'N/A')
        if answer != 'N/A':
            translated_answer = self.translate_text(answer)
            self.answer_label.config(text=f"Answer: {translated_answer}")
            # messagebox.showinfo("Translation", "Answer translation completed!")

if __name__ == "__main__":
    root = tb.Window(themename="cosmo")
    app = AlignApp(root)
    root.mainloop()
